Ext.namespace('Ext.ux.plusmpm.scheduledtasks');

Ext.ux.plusmpm.scheduledtasks.ParametersBuilder = function (config) {
  Ext.applyIf(config, {
    parametersDefinition: new Ext.util.MixedCollection(),
  });

  Ext.ux.plusmpm.scheduledtasks.ParametersBuilder.superclass.constructor.call(this, config);
};

Ext.extend(Ext.ux.plusmpm.scheduledtasks.ParametersBuilder, Ext.Component, {
  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.ParametersBuilder.superclass.initComponent.call(this);
  },

  isComponentParameter: function (id) {
    return this.parametersDefinition.contains(id);
  },

  eachParam: function (func, scope) {
    this.parametersDefinition.eachKey(func, scope);
  },

  createParameter: function (paramDefinitionWithConfig, singularFieldExtraConfig) {
    var listeners = null;

    if (singularFieldExtraConfig) {
      listeners = singularFieldExtraConfig.listeners;
      delete singularFieldExtraConfig.listeners;
    }

    const field = this.createField(paramDefinitionWithConfig, singularFieldExtraConfig);

    if (listeners) {
      var fieldListeners = field.listeners;

      var mergedListeners = PW.ScheduledTaskUtils.getMergedListeners(fieldListeners, listeners);
      if (mergedListeners) {
        field.listeners = mergedListeners;
      }
    }

    return field;
  },

  createField: function (paramDefinitionWithConfig, singularFieldExtraConfig) {
    var fieldType = paramDefinitionWithConfig.fieldType;

    var field;
    switch (fieldType) {
      case 'string[]':
      case 'string': {
        field = this.createTextField();
        break;
      }
      case 'integer[]':
      case 'integer': {
        field = this.createNumberField(false);
        break;
      }
      case 'float[]':
      case 'float': {
        field = this.createNumberField(true);
        break;
      }
      case 'date[]':
      case 'date': {
        field = this.createDateField();
        break;
      }
      case 'datetime[]':
      case 'datetime': {
        field = this.createDateTimeField();
        break;
      }
      case 'boolean[]':
      case 'boolean': {
        field = this.createBooleanField();
        break;
      }
      case 'file[]':
      case 'file': {
        field = this.createFileField();
        break;
      }
      default: {
        field = this.createTextField();
        break;
      }
    }

    if (singularFieldExtraConfig) {
      Ext.apply(field, singularFieldExtraConfig);
    }

    const isArray = paramDefinitionWithConfig.fieldType.indexOf('[]') !== -1;
    const paramData = this.createParamData(paramDefinitionWithConfig, isArray);

    if (isArray) {
      var arrayField = {
        xtype: 'st_arrayfield',
        renderButtons: true,

        fieldConfig: field,
        paramData: paramData,
      };

      Ext.apply(arrayField, paramData.apiInfo);

      arrayField = this.setLabel(arrayField, paramDefinitionWithConfig);
      arrayField = this.addDescription(arrayField, paramDefinitionWithConfig);

      return arrayField;
    } else {
      const fieldListeners = field.listeners;
      const paramListeners = paramData.apiInfo.listeners;

      Ext.apply(field, paramData.apiInfo);

      const mergedListeners = PW.ScheduledTaskUtils.getMergedListeners(fieldListeners, paramListeners);
      if (mergedListeners) {
        field.listeners = mergedListeners;
      }

      if (field.requiresParamData) {
        Ext.apply(field, {
          paramData: paramData,
        });
      } else {
        Ext.apply(field, paramData.valueInfo);
        Ext.apply(field, paramData.fieldInfo);
      }

      field = this.setLabel(field, paramDefinitionWithConfig);
      field = this.addDescription(field, paramDefinitionWithConfig);
      return field;
    }
  },

  createParamData: function (paramDefinitionWithConfig, isArray) {
    var valueInfo;
    if (isArray) {
      valueInfo = [];
      if (paramDefinitionWithConfig.value !== null && paramDefinitionWithConfig.value !== undefined) {
        const values = paramDefinitionWithConfig.value;
        for (var i = 0; i < values.length; ++i) {
          let value = values[i] == null ? undefined : values[i];

          var singleValueInfo = {
            value: value,
            checked: Ext.isString(value) ? value === 'true' : !!value,
          };

          if (paramDefinitionWithConfig.valueMetadata) {
            Ext.apply(singleValueInfo, {
              valueMetadata: paramDefinitionWithConfig.valueMetadata[i],
            });
          }

          valueInfo.push(singleValueInfo);
        }
      }
    } else {
      let value = paramDefinitionWithConfig.value == null ? undefined : paramDefinitionWithConfig.value;

      valueInfo = {
        value: value,
        checked: Ext.isString(value) ? value === 'true' : !!value,
      };

      if (paramDefinitionWithConfig.valueMetadata) {
        Ext.apply(valueInfo, {
          valueMetadata: paramDefinitionWithConfig.valueMetadata[0],
        });
      }
    }

    return {
      apiInfo: {
        id: paramDefinitionWithConfig.componentParamId,
        listeners: paramDefinitionWithConfig.listeners,
      },
      valueInfo: valueInfo,
      fieldInfo: {
        componentParameterId: paramDefinitionWithConfig.componentParamId,
        allowBlank: !!paramDefinitionWithConfig.optional,
        height: 40,
        submitValue: false,
        name: Ext.id(null, 'param_' + paramDefinitionWithConfig.position + '_'),
        hiddenName: Ext.id(null, 'param_' + paramDefinitionWithConfig.position + '_'),
        isScheduledTaskParameter: true,
        parameterPosition: paramDefinitionWithConfig.position,
      },
    };
  },

  setLabel: function (field, paramDefinition) {
    this.refreshLabel(null, field, paramDefinition);

    return field;
  },

  refreshLabel: function (parameterContainer, field, paramDefinition) {
    var labelText = this._generateLabelText(paramDefinition);
    const descIcon = ' <span class="tooltip-image" style="white-space: pre;">       </span>';

    const container = parameterContainer;
    const isInInnerParameterContainer = container && container.isInnerContainer;

    if (!isInInnerParameterContainer && !Ext.isEmpty(field.description)) {
      labelText += descIcon;
    }

    this._setLabelText(field, labelText);

    if (isInInnerParameterContainer) {
      var containerLabelText = container.initialLabel;
      if (!containerLabelText) {
        const labels = [];
        var renderDesc = false;

        Ext.each(
          container.items.items,
          function (cmp) {
            labels.push(cmp.fieldLabel);
            if (!Ext.isEmpty(cmp.description)) {
              renderDesc = true;
            }
          },
          container
        );

        const cleanedLabels = Ext.clean(labels);
        if (cleanedLabels.length > 0) {
          containerLabelText = cleanedLabels.join(', ');

          if (renderDesc) {
            containerLabelText += descIcon;
          }

          container.labelSeparator = ':';
        }
      }

      this._setLabelText(container, containerLabelText);
    }
  },

  _generateLabelText: function (paramDefinition) {
    const type = paramDefinition.componentParamId ? '' : ' (' + paramDefinition.type + ')';

    if (paramDefinition.optional) {
      return paramDefinition.name + type;
    } else {
      return paramDefinition.name + type + '<span class="form-field-required-sign">*</span>';
    }
  },

  addDescription: function (field, paramDefinition) {
    field.description = paramDefinition.description;

    var me = this;
    const afterRenderListener = {
      afterrender: function (field) {
        if (Ext.isEmpty(this.description)) {
          return;
        }

        var label = me._findClosestLabel(field) || field.el;

        if (!label.descriptionToolTip) {
          label.descriptionToolTip = new Ext.ux.plusmpm.DescriptionToolTip({
            html: this.description,
            target: label,
          });
        } else {
          var toolTip = label.descriptionToolTip;

          if (toolTip.body) {
            toolTip.body.dom.innerHTML += '<br><br>' + this.description;
          } else {
            toolTip.html += '<br><br>' + this.description;
          }
        }
      },
    };

    var mergedListeners = PW.ScheduledTaskUtils.getMergedListeners(field.listeners, afterRenderListener);
    if (mergedListeners) {
      field.listeners = mergedListeners;
    }

    return field;
  },

  addDescriptionTooltipTo: function (target, description) {
    target.description = description;

    const afterRenderListener = {
      afterrender: function (component) {
        if (Ext.isEmpty(this.description)) {
          return;
        }

        if (!component.descriptionToolTip) {
          component.descriptionToolTip = new Ext.ux.plusmpm.DescriptionToolTip({
            html: this.description,
            target: component.el,
          });
        } else {
          var toolTip = component.descriptionToolTip;

          if (toolTip.body) {
            toolTip.body.dom.innerHTML += '<br><br>' + this.description;
          } else {
            toolTip.html += '<br><br>' + this.description;
          }
        }
      },
    };

    var mergedListeners = PW.ScheduledTaskUtils.getMergedListeners(target.listeners, afterRenderListener);
    if (mergedListeners) {
      target.listeners = mergedListeners;
    }

    return target;
  },

  _findClosestLabel: function (field) {
    var label = field.label;

    if (!label) {
      const compWithLabel = field.findParentBy(function (container) {
        return !!container.label;
      });

      label = compWithLabel ? compWithLabel.label : null;
    }

    return label;
  },

  _setLabelText: function (field, labelText) {
    field.fieldLabel = labelText;

    if (field.label) {
      field.label.update(field.fieldLabel + (field.labelSeparator || ''));
    }
  },

  createTextField: function () {
    return {
      xtype: 'textfield',

      requiresParamData: false,
    };
  },

  createNumberField: function (floatingPoint) {
    const field = {
      xtype: 'numberfield',
      allowDecimals: floatingPoint,

      requiresParamData: false,
    };

    if (floatingPoint) {
      field.decimalPrecision = 8;
    }

    return field;
  },

  createDateField: function () {
    return {
      xtype: 'datefield',
      listeners: {
        invalid: function (field) {
          field.wrap.addClass('x-form-invalid');
        },

        valid: function (field) {
          field.wrap.removeClass('x-form-invalid');
        },
      },
      invalidClass: 'avoid-border-non-existent-class',

      requiresParamData: false,
    };
  },

  createBooleanField: function () {
    return {
      xtype: 'checkbox',
      inputValue: 'true',

      requiresParamData: false,
    };
  },

  createDateTimeField: function () {
    return {
      xtype: 'st_datetimefield',

      requiresParamData: true,
    };
  },

  createFileField: function () {
    return {
      xtype: 'st_filefield',

      requiresParamData: true,
    };
  },

  _readDefinition: function (config) {
    if (Ext.isString(config)) {
      config = {
        id: config,
      };
    }
    const definition = this.parametersDefinition.get(config.id);
    if (!definition) {
      return {};
    }

    return Ext.apply({}, definition, config);
  },
});

Ext.reg('st_parametersbuilder', Ext.ux.plusmpm.scheduledtasks.ParametersBuilder);
