(function(Ext) {

	Ext.define('Suncode.datasourcemanager.Manager', {
		extend : 'Ext.app.Controller',

		stores : [ 'Suncode.datasourcemanager.store.DataSourceStore',
				'Suncode.datasourcemanager.store.DataSourceTypeStore',
				'Suncode.datasourcemanager.store.DataSourceCategoryStore'],
		models : [ 'Suncode.datasourcemanager.model.DataSourceTreeNode',
				'Suncode.datasourcemanager.model.DataSource',
				'Suncode.datasourcemanager.model.DataSourceType',
				'Suncode.datasourcemanager.model.DataSourceCategory'],
		views : [ 'Viewport', 'Suncode.datasourcemanager.view.DataSourceTree',
				'Suncode.datasourcemanager.view.DataSourceConfiguration',
				'Suncode.datasourcemanager.component.MultiValueField',
				'Suncode.datasourcemanager.component.GroupedMultiValueField',
				'Suncode.datasourcemanager.component.RemovableField',
				'Ext.ux.DateTimeField',
				'Suncode.datasourcemanager.component.ExtendedDateTimeField',
				'Suncode.datasourcemanager.mixins.TreeFilter',
				'Suncode.datasourcemanager.mixins.ToolTip',
				'Suncode.datasourcemanager.component.JsonImportWindow',
				'Suncode.datasourcemanager.component.ExecutionParametersWindow',
				'Suncode.datasourcemanager.component.DataSourceResultWindow'],

		refs : [ {
			ref : 'dataSourceTree',
			selector : 'data-source-tree'
		}, {
			ref : 'configurationForm',
			selector : 'data-source-configuration'
		}, {
			ref : 'createBtn',
			selector : '#create-data-source-button'
		}, {
			ref : 'saveBtn',
			selector : '#save-data-source-button'
		}, {
			ref : 'removeBtn',
			selector : '#remove-data-source-button'
		}, {
            ref : 'copyBtn',
            selector : '#copy-data-source-button'
        }, {
            ref : 'executeBtn',
            selector : '#execute-data-source-button'
        }],

		init : function() {
			var me = this;

			me.eventbus.unlisten(me.id);
			me.control({
				'data-source-tree' : {
					selectdatasource : me.onSelectDataSource
				},
				'data-source-configuration' : {
					afterrender : me.buildEmptyConfiguration
				},
				'#new-data-source-button' : {
					click : me.onNewButtonClick
				},
				'#export-all-data-sources-button' : {
                    click : me.onExportAllButtonClick
                },
                '#export-selected-data-sources-button' : {
                    click : me.onExportSelectedButtonClick
                },
                '#import-data-sources-button' : {
                    click : me.onImportButtonClick
                },
                'jsonimportwindow' : {
                    'jsonimport' : me.doJsonImport
                },
				'#create-data-source-button' : {
					click : me.onCreateButtonClick
				},
				'#save-data-source-button' : {
					click : me.onSaveButtonClick
				},
				'#remove-data-source-button' : {
					click : me.onRemoveButtonClick
				},
				'#copy-data-source-button' : {
				    click : me.onCopyButtonClick
				},
				'#execute-data-source-button' : {
				    click : me.onExecuteButtonClick
				},
				'executionparameterswindow' : {
                    'executeDataSource' : me.executeDataSource,
                    close: me.onTemporaryWindowClose
                },
                'datasourceresultwindow': {
                    close: me.onTemporaryWindowClose
                },
				'#datasource-type-combo' : {
					select: me.onDataSourceTypeComboSelect
				},
				'#datasource-operation-combo' : {
					change: me.onDataSourceOperationComboSelect
				},
				'#treeFilter': {
					change: me.filter
				},
				'#datasource-category-combo-filter':{
					change: me.filter
				},
				'#datasource-type-allow_cache': {
					change: me.onAllowCacheChange
				}
			});
		},

		onSelectDataSource : function(record) {
			this.getConfigurationForm().mask();
			Ext.defer(function(){
				if(record) {
					if(record.get('leaf')) {
						this.selectDataSource(record);
					}
					else {
						var dataSourceType = this._getDataSourceId( record );
						if(dataSourceType && !record.get('category')) {
							this.buildDataSourceConfiguration( dataSourceType );
						}
						else {
							this.buildEmptyConfiguration();
							this.getConfigurationForm().unmask();
						}
					}
				}
			},1, this);
		},

		selectDataSource : function(record) {
			var form = this.getConfigurationForm(),
				id = this._getDataSourceId( record );

			form.mask(PW.t('loading-data-source'));
			this.loadDataSourceDeclaration(id);
		},

		_getDataSourceId: function(record){
			var id = record.get('id');
			return id.split('#')[1];
		},

		buildEmptyConfiguration: function() {
			Ext.suspendLayouts();

			var form = this.getConfigurationForm();
		    form.setTitle( PW.t('data-source-configuration-new') );
		    form.getForm().reset();
			form.setReadOnlyField('id', false);
			form.setReadOnlyField('type', false);
			form.setReadOnlyField('operation', false);
			form.clearOperations();
			form.removeAllParameters();

			this.getCreateBtn().show();
			this.getSaveBtn().hide();
			this.getRemoveBtn().hide();
			this.getCopyBtn().hide();

			Ext.resumeLayouts(true);
		},

		buildDataSourceConfiguration: function( dataSourceType ) {
			Ext.suspendLayouts();

			this.buildEmptyConfiguration();
			this.loadDataSourceConfiguration( dataSourceType );
		},

		buildDataSourceInstanceConfiguration: function( record ) {
			Ext.suspendLayouts();

			var form = this.getConfigurationForm(),
				dataSourceType = record.get('type');
			this.setEditConfigurationView(form);
			this.loadDataSourceConfiguration( dataSourceType, record );
		},

		setEditConfigurationView: function(form)
		{
			form.setTitle( PW.t('data-source-configuration-edit') );
			form.setReadOnlyField('id', true);
			form.setReadOnlyField('type', true);

			this.getCreateBtn().hide();
			this.getSaveBtn().show();
			this.getRemoveBtn().show();
			this.getCopyBtn().show();
		},

		loadDataSourceConfiguration: function( dataSourceType, record ){
			var me = this;

			Ext.Ajax.request({
				scope: this,
				url: Suncode.getAbsolutePath('/api/datasources/definition/') + dataSourceType,
				success: function( response ) {
					if(response.responseText == ""){
						PW.ui.Message.error(PW.t('loading-data-source-type-error'));
						return;
					}

					var component = Ext.decode(response.responseText),
					    form = me.getConfigurationForm();

					form.componentDefinition = component;
					form.setType( component );

					if (!form.getAllowCache()) {
						form.setCacheExpiredVisible(false)
					}

					form.setOperationType( component.operations );
					if(record) {
						form.setReadOnlyField('type', true);
						form.buildParameters();
					    form.loadRecord(record);
					}
				},
				failure: function() {
					PW.ui.Message.error(PW.t('loading-data-source-type-error'));
				},
				callback: function(){
					me.getConfigurationForm().unmask();
					Ext.resumeLayouts(true);
				}
			});
		},

		buildDataSourceCopyConfiguration: function() {
			Ext.suspendLayouts();

			var form = this.getConfigurationForm();
			form.setTitle(PW.t('data-source-configuration-new'));
			form.setReadOnlyField('id', false);
			form.getForm().setValues({
                id: '',
                name: ''
            });

			this.getCreateBtn().show();
			this.getSaveBtn().hide();
			this.getRemoveBtn().hide();
			this.getCopyBtn().hide();

			Ext.resumeLayouts(true);
		},

		onNewButtonClick : function() {
			this.buildEmptyConfiguration();
		},

		onExportAllButtonClick: function() {
		    var me = this;
            this.getConfigurationForm().mask();

            Ext.Ajax.request({
                scope: this,
                url: Suncode.getAbsolutePath('/api/datasources/declarations/export/all'),
                method: 'POST',
                success: function(response) {
                    window.location = PW.getAbsolutePath('api/datasources/declarations/download/' + Ext.decode(response.responseText));
                },
                failure: function() {
                    PW.ui.Message.error(PW.t('loading-data-source-type-error'));
                },
                callback: function(){
                    me.getConfigurationForm().unmask();
                }
            });
        },

		onExportSelectedButtonClick: function() {
		    this.getConfigurationForm().mask();

		    var me = this,
		        toExport = [],
		        tree = this.getDataSourceTree(),
		        selected = tree.getSelectionModel().getSelection();

		    if(!selected || selected.length == 0) {
		        PW.ui.Message.error(PW.t('datasources-nothing-selected-message'));
		        this.getConfigurationForm().unmask();
		        return;
		    }

		    Ext.each(selected, function( node ) {
		        if(node.isLeaf()) {
		            toExport.push({
		                id: this._getDataSourceId(node),
		                definition: false
		            });
		        }
		        else {
		        	if(this._isCategory(node)){
		        		Ext.each(node.childNodes, function(child){
		        			toExport.push({
		                        id: this._getDataSourceId(child),
		                        definition: true
		                    });
		        		}, this);
		        	}
		        	else {
		        		toExport.push({
		        			id: this._getDataSourceId(node),
		        			definition: true
		        		});
		        	}
		        }
		    }, this);

		    Ext.Ajax.request({
                scope: this,
                url: Suncode.getAbsolutePath('/api/datasources/declarations/export'),
                method: 'POST',
                jsonData: toExport,
                success: function(response) {
                    window.location = PW.getAbsolutePath('api/datasources/declarations/download/' + Ext.decode(response.responseText));
                },
                failure: function() {
                    PW.ui.Message.error(PW.t('loading-data-source-type-error'));
                },
                callback: function(){
                    me.getConfigurationForm().unmask();
                }
            });
		},

		_isCategory: function(node){
			return Ext.isEmpty(node.get('id'));
		},

		onImportButtonClick: function() {
		    new Suncode.datasourcemanager.component.JsonImportWindow().show();
        },

        doJsonImport: function(importWindow, form) {
            var me = this;
			const store = this.getConfigurationForm().getForm().findField('category').getStore()
            if( form.isValid() ){
                form.submit({
                    scope: this,
                    url: PW.getAbsolutePath('api/datasources/declarations/import'),
                    waitMsg: PW.t('progressTitle'),
                    failure: function(form, action) {
                        var response = Ext.decode( action.response.responseText );
                        if(response.message) {
                            PW.ui.Message.error( response.message );
                        }
                        else {
                            PW.ui.Message.error( PW.t('datasources-import-failure') );
                        }
                    },
                    success: function(form, action) {
                        var response = Ext.decode( action.response.responseText );
                        var message = response.message.replaceAll("[br]", "</br>");
                        PW.ui.Message.success( message );
                        me.getDataSourceTree().refresh();
						store.reload()
                        importWindow.close();
                    }
                });
            }
        },

		onCreateButtonClick : function() {
			this.saveForm(true);
		},

		onSaveButtonClick : function() {
			this.saveForm();
		},

		saveForm: async function(isNew) {
			var form = this.getConfigurationForm();

			if(form.isValid()) {
				var declaration = await this.buildDeclaration( form, true );

				Ext.Ajax.request({
					scope: this,
					url: Suncode.getAbsolutePath('/api/datasources/declaration/' + (isNew ? 'create' : 'update')),
					params: {form: Ext.encode( declaration )},
					success: function(responseObj) {
						var response = Ext.decode(responseObj.responseText);

						if(isNew || form.shouldRefresh()){
							this.refreshTree({id: declaration.id, name: declaration.name, type: declaration.type, category: declaration.category });
						}
						else {
							this.updateTree({id: declaration.id, name: declaration.name});
						}

						if(response.success){
							PW.ui.Message.success(response.message);
							if(isNew)
							{
								this.loadDataSourceDeclaration(declaration.id)
								this.setEditConfigurationView(form);
							}

							form.getForm().findField('category').getStore().reload()
						}
						else {
							PW.ui.Message.error(response.message);
						}
					},
					failure: function() {
						PW.ui.Message.error(PW.t('saving-data-source-error'));
					},
					callback: function(){
						form.unmask();
					}
				}, this);
			}
		},

		loadDataSourceDeclaration: function(declarationId) {
			var me = this;
			var DataSource = Ext.ModelManager.getModel('Suncode.datasourcemanager.model.DataSource');

			DataSource.load(declarationId, {
				scope: this,
				success: function(record){
					me.buildDataSourceInstanceConfiguration(record);
				},
				failure: function(){
					PW.ui.Message.error(PW.t('loading-data-source-error'));
				}
			});
		},

		buildDeclaration: async function( configurationPanel, createCategory ) {
		    const me = this
			const form = configurationPanel.getForm()
			const category = await this.getCategory(form, createCategory);

			const declaration = {
				type: form.findField('type').getValue(),
				id: form.findField('id').getValue(),
				name: form.findField('name').getValue(),
				description: form.findField('description').getValue(),
				operation : form.findField('operation').getValue(),
				replaceSemicolonsChar : form.findField('replaceSemicolonsChar').getValue(),
				allowCache : form.findField('allowCache').getValue(),
				cacheExpirationDelaySec : form.findField('cacheExpirationDelaySec').getValue(),
				category: category,
			}
			const parameters = {};

		    configurationPanel.getParameters().eachKey(function( key, parameter ) {
		        parameters[key] = me.buildParameterDeclaration(form, key, parameter);
		    });

		    declaration.parameters = parameters;
		    return declaration;
		},

		getCategory(form, createCategory) {
			return new Promise((resolve, reject) => {
				const categoryValue = form.findField('category').getValue();
				if (!categoryValue || !isNaN(categoryValue) || !createCategory) {
					return resolve(categoryValue);
				}

				Ext.Ajax.request({
					url: Suncode.getAbsolutePath('/api/datasource/category/'),
					method: 'POST',
					params: {
						name: categoryValue,
					},
					success:  (result, action, response) => {
						form.findField('category').getStore().reload()
						resolve(Ext.decode(result.responseText))
					},
					failure:(response) => {
						reject(response)
					}
				});
			});
		},

		buildParameterDeclaration: function(form, key, parameter) {
		    var paramCmp = form.findField(key),
		        value;
            if(!paramCmp) {
                return this.handleNonExistingParameter(parameter);
            }
            if(paramCmp.isXType('multivaluefield')) {
                value = paramCmp.getValue( key );
            }
            else {
                var parent = paramCmp.ownerCt.ownerCt.ownerCt;
                if( parent.isXType('groupedmultivaluefield') ) {
                    value = parent.getValue( key );
                }
                else {
                    value = paramCmp.getValue();
                }
            }

            if( parameter.array ) {
                var param = [];
                Ext.each(value, function(val) {
                    param.push({
                        value: val,
                        type: parameter.type
                    });
                });

                return param;
            }

            return {
                value: value,
                type: parameter.type
            }
		},

		handleNonExistingParameter: function(parameter) {
		    if(parameter.array) {
                return [];
            }
            return {
                value: null,
                type: parameter.type
            };
		},

		refreshTree: function(data){
			this.getDataSourceTree().refresh(data);
		},

		updateTree: function(data){
			this.getDataSourceTree().updateSelected(data);
		},

		onRemoveButtonClick: function() {
			var form = this.getConfigurationForm(),
				record = form.getRecord();
			const me = this

			form.mask(PW.t('removing-data-source'));

			Ext.Msg.confirm(PW.t('remove-data-source'), PW.t('remove-data-source-question', record.get('name')), function(btn){
				if(btn == 'yes') {
					if(record)
					{
						const tree = this.getDataSourceTree();
						const parentId = tree.getParentId(record.get('id'))
						record.destroy({
							scope: this,
							success: function() {
								tree.getStore().reload({
									callback: function() {
										me.filter()
										tree.expandPathById(parentId);
									}
								});
								form.getForm().findField('category').getStore().reload()
								this.buildEmptyConfiguration();
								PW.ui.Message.success(PW.t('removing-data-source-success'));
							},
							failure: function() {
								PW.ui.Message.error(PW.t('removing-data-source-error'));
							},
							callback: function() {
								form.unmask();
							}
						});
					}
				}
				else if(btn == 'no' || btn == 'cancel') {
					form.unmask();
				}
			}, this);
		},

		onCopyButtonClick: function() {
		    this.buildDataSourceCopyConfiguration();
		},

		onExecuteButtonClick: async function() {
		    var form = this.getConfigurationForm();

            if( form.isValid() )
            {
    		    Ext.Ajax.request({
                    scope: this,
                    url: Suncode.getAbsolutePath('/api/datasources/temporary/create'),
                    method: 'POST',
                    params: {
                        form: Ext.encode( await this.buildDeclaration( form, false ) )
                    },
                    success: function(response) {
                        var resp = Ext.decode( response.responseText );
                        if(resp.success) {
                            var temporaryDataSource = Ext.decode( response.responseText ).entity;
                            new Suncode.datasourcemanager.component.ExecutionParametersWindow( temporaryDataSource ).show();
                        }
                        else {
                            PW.ui.Message.error( resp.message );
                        }
                    },
                    failure: function() {
                        var message = Ext.decode( response.responseText ).message;
                        PW.ui.Message.error( message );
                    },
                    callback: function(){
                        form.unmask();
                    }
                }, this);
            }
		},

		executeDataSource: function(win, form) {
		    if( form.isValid() ) {
		        var operation = win.temporaryDataSource.dataSource.declaration.operation;
		        form.submit({
                    scope: this,
                    url: PW.getAbsolutePath('api/datasources/temporary/' + win.temporaryDataSource.uuid + '/' + operation.toLowerCase()),
                    headers: {
                        //musi być, bo w przypadku braku pól na formie wysyłany jest zły contentType.
                        'Content-Type': 'application/x-www-form-urlencoded'
                    },
                    waitMsg: PW.t('progressTitle'),
                    failure: function(form, action) {
                        var response = Ext.decode( action.response.responseText );
                        win.close();
                        PW.ui.Message.warning( response.message, { hideAfter: 0, showCopyButton: true });
                    },
                    success: function(form, action) {
                        if(operation == 'READ') {
                            win.notInvalidateDataSource = true;
                            win.close();
                            new Suncode.datasourcemanager.component.DataSourceResultWindow( win.temporaryDataSource ).show();
                        }
                        else {
                            PW.ui.Message.success( PW.t('datasources-operation-success') );
                        }
                    }
                });
		    }

		},

		onDataSourceTypeComboSelect: function(combo, record) {
			Ext.suspendLayouts();
			this.getConfigurationForm().mask();
			record = Ext.isArray(record) ? record[0] : record;
			this.loadDataSourceConfiguration( record.get('id') );
		},

		onDataSourceOperationComboSelect: function(combo, record) {
			if( combo.getValue() ) {
				Ext.suspendLayouts();
				var form = this.getConfigurationForm();
				form.mask();
				try {
					form.buildParameters();
				} finally {
					form.unmask();
					Ext.resumeLayouts(true);
				}
			}
		},

		filter: function(){
			const form =  this.getDataSourceTree().down('#filterForm').getForm()
			const query = form.findField("query").getValue();
			const category = form.findField("category").getValue();
			this.getDataSourceTree().filter(query, category);
		},

		onAllowCacheChange: function(filterField, newValue){
			this.getConfigurationForm().setCacheExpiredVisible(newValue)
		},


		onTemporaryWindowClose: function(win) {
		    if(win.notInvalidateDataSource) {
		        return;
		    }
		    Ext.Ajax.request({
                scope: this,
                url: PW.getAbsolutePath('api/datasources/temporary/' + win.temporaryDataSource.uuid + '/remove'),
                method: 'DELETE'
            }, this);
	    }
	});

}(Ext4));
