Ext.namespace('Ext.ux.plusmpm.scheduledtasks');

/**
 * Bottom Toolbar zadań zaplanowanych.
 * Umożliwia kontrole stanu wykonywania zadania:
 * usuwanie, uruchamianie, anulowanie, podgląd historii i logów
 *
 * config.task - TaskPanel skojarzony z tym toolbarem
 */
Ext.ux.plusmpm.scheduledtasks.TaskControlToolbar = function (config) {
  this.updateCategoryStore();

  var activeText = PW.t('active');
  var noactiveText = PW.t('notActive');
  var task = config.taskRecord.data;

  config = Ext.apply(
    {
      taskId: task.id,
      items: [
        /**
         * Status zadania
         */
        {
          xtype: 'splitbutton',
          ref: 'statusSplitButton',
          iconAlign: 'right',
          width: 100,
          menu: new Ext.menu.Menu({
            items: [
              { text: PW.t('deactivateTask'), handler: this.onDeactivateHandler, scope: this, hidden: !task.active, ref: 'deactivateMenu' },
              {
                text: PW.t('activateTask'),
                handler: this.onActivateHandler,
                scope: this,
                hidden: task.active,
                disabled: !task.available,
                ref: 'activateMenu',
              },
            ],
          }),
          handler: function () {
            this.showMenu();
          },
          iconCls: task.active ? 'dvnt-icon-check-circle dvnt-green-icon' : 'dvnt-icon-x-circle dvnt-red-icon',
          text: task.active ? activeText : noactiveText,
          activeText: activeText,
          noactiveText: noactiveText,
          updateStatus: function (active, processing, progress) {
            if (active) {
              this.setIconClass('dvnt-icon-check-circle dvnt-green-icon');
              this.setText(activeText);
              this.menu.activateMenu.hide();
              this.menu.deactivateMenu.show();
            } else {
              this.setIconClass('dvnt-icon-x-circle dvnt-red-icon');
              this.setText(noactiveText);
              this.menu.activateMenu.show();
              this.menu.deactivateMenu.hide();
            }
          },
        },
        '-',
        PW.t('status') + ': ',
        {
          xtype: 'tbtext',
          text: (function () {
            if (task.available) {
              return task.processing ? PW.t('running') : PW.t('stopped');
            } else {
              var info = PW.t('Zadanie_nie_jest_dostepne');
              if (task.pluginId) {
                info += ' (' + PW.t('plugin') + ': <b>' + task.pluginId + '</b>)';
              }
              return info;
            }
          })(),
          ref: 'processingTbItem',
        },

        /**
         * Progress wykonywania
         */
        {
          xtype: 'container',
          ref: 'progressContainer',
          hidden: task.progress <= -1,
          style: 'padding-top: 2px; ',
          defaults: {
            style: 'float: left;',
          },
          items: [
            {
              xtype: 'tbseparator',
            },
            {
              xtype: 'tbtext',
              text: PW.t('progress') + ': ',
            },
            task.progress <= 0
              ? {
                  xtype: 'tbtext',
                  text: PW.t('W_trakcie_wykonywania'),
                }
              : {
                  xtype: 'progress',
                  width: 100,
                  text: `${Math.round(100 * task.progress)}%`,
                  value: task.progress,
                  ref: 'progressbar',
                  style: {
                    cursor: 'pointer',
                    float: 'left',
                  },
                  animate: true,
                  listeners: {
                    afterRender: function (progress) {
                      progress.el.on('click', this.oProgressClick, this);
                    },
                    scope: this,
                  },
                },
          ],
        },

        /**
         * Zmiana kategorii
         */
        '->',
        {
          xtype: 'splitbutton',
          width: 100,
          menu: new Ext.menu.Menu({
            items: [
              {
                xtype: 'combo',
                ref: 'categoryComboChange',
                emptyText: PW.t('unassigned'),
                store: this.categoryStore,
                triggerAction: 'all',
                mode: 'local',
                displayField: 'name',
                valueField: 'name',
                enableKeyEvents: true,
                listeners: {
                  scope: this,
                  select: this.onCategoryChange,
                  specialkey: function (field, e) {
                    if (field.isValid() && e.getKey() == e.ENTER) {
                      var name = field.getRawValue();
                      this.onCategoryChange(field, { data: { name: name } });
                    }
                  },
                },
              },
              {
                xtype: 'button',
                anchor: '100%',
                text: PW.t('changeCategory'),
                scope: this,
                style: 'padding: 5px; margin-top: 5px;',
                handler: function (field) {
                  if (field.ownerCt.categoryComboChange.isValid()) {
                    var name = field.ownerCt.categoryComboChange.getRawValue();
                    this.onCategoryChange(field, { data: { name: name } });
                  }
                },
              },
            ],
          }),
          handler: function () {
            this.showMenu();
          },
          iconCls: 'dvnt-icon-edit-form',
          text: PW.t('changeCategory'),
        },
        '-',

        /**
         * Sterowanie zadaniem
         */
        {
          xtype: 'button',
          disabled: !task.available || task.processing,
          text: PW.t('edit'),
          handler: this.onEditClick,
          iconCls: 'dvnt-icon-settings',
          ref: 'editButton',
          scope: this,
        },
        '-',
        {
          xtype: 'button',
          disabled: !task.available,
          text: PW.t('execute'),
          hidden: task.processing,
          handler: this.onExecuteClick,
          iconCls: 'dvnt-icon-right',
          ref: 'executeButton',
          scope: this,
        },
        {
          xtype: 'button',
          text: PW.t('cancelExecution'),
          hidden: !task.processing,
          handler: this.onCancelClick,
          disabled: !task.cancelable,
          iconCls: 'dvnt-icon-x-symbol',
          ref: 'cancelButton',
          scope: this,
        },
        {
          xtype: 'button',
          text: PW.t('logs'),
          handler: this.onHistoryClick,
          iconCls: 'dvnt-icon-time',
          scope: this,
        },
        {
          xtype: 'button',
          text: PW.t('delete'),
          handler: this.onDeleteClick,
          iconCls: 'dvnt-icon-delete',
          scope: this,
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.TaskControlToolbar.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.TaskControlToolbar constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.TaskControlToolbar, Ext.Toolbar, {
  updateCategoryStore: function () {
    var categoryStore = Ext.StoreMgr.get('tasksCategoryStore');
    var categories = [];

    categoryStore.each(function (item) {
      categories.push([item.data.id, item.data.name]);
    }, this);

    if (this.categoryStore) {
      this.categoryStore.loadData(categories);
    } else {
      this.categoryStore = new Ext.data.ArrayStore({
        autoDestroy: true,
        data: categories,
        fields: ['id', 'name'],
      });
    }
  },

  onCategoryChange: function (combo, record) {
    var store = Ext.StoreMgr.get('tasksCategoryStore');
    var index = store.findExact('name', record.data.name);
    if (index > -1) {
      record = store.getAt(index);
    }

    this.requestCategryChange(record.data.id, record.data.name, combo.ownerCt);
  },

  /**
   * Obsługa aktywacji / deaktywacji zadania
   */
  onDeactivateHandler: function () {
    this.onActiveToggle(false);
  },

  onActivateHandler: function () {
    this.onActiveToggle(true);
  },

  onActiveToggle: function (active) {
    this.requestActiveToggle(active);
  },

  onActiveToggleSuccess: function (active) {
    this.statusSplitButton.updateStatus(active);
    this.fireEvent('activechange', active);
  },

  onActiveToggleFailure: function (active) {
    //jakaś wiadomość, informacja.
  },

  /**
   * Zdarzenie kliknięcia przycisku edycji zadania zaplanowanego
   */
  onEditClick: function (button, event) {
    this.fireEvent('beforeparametersrender');

    new Ext.ux.plusmpm.scheduledtasks.EditTaskWindow({
      taskRecord: this.taskRecord,
    }).show();
  },

  /**
   * Zdarzenie kliknięcia na pasek postępu
   */
  oProgressClick: function (button, event) {
    this.fireEvent('history', this, true);
  },

  /**
   * Zdarzenie kliknięcia przycisku wyświetlenia logów
   */
  onHistoryClick: function (button, event) {
    this.fireEvent('history', this, false);
  },

  /**
   * Zdarzenie kliknięcia przycisku wyświetlenia logów
   */
  onDeleteClick: function (button, event) {
    Ext.Msg.show({
      title: PW.t('deletingTask'),
      msg: PW.t('sureToDeleteTask'),
      buttons: Ext.Msg.YESNO,
      fn: function (buttonId) {
        if (buttonId == 'yes') {
          this.fireEvent('remove', this);
        }
      },
      scope: this,
      icon: Ext.MessageBox.QUESTION,
    });
  },

  /**
   * Zdarzenie kliknięcia przycisku anuluj zadanie
   */
  onCancelClick: function (button, event) {
    this.fireEvent('cancel', this);
  },

  /**
   * Zdarzenie kliknięcia przycisku wykonaj zadanie
   */
  onExecuteClick: function (button, event) {
    this.fireEvent('execute', this);
  },

  /**
   * Zdarzenie zmiany stanu wykonywania zadania
   */
  onTaskUpdate: function (record) {
    this.taskRecord = record;
    var data = record.data;

    var active = data.active;
    var progress = data.progress;
    var processing = data.processing;
    var available = data.available;
    var pluginId = data.pluginId;

    this.updateCategoryStore();

    this.statusSplitButton.updateStatus(active);
    this.processingTbItem.setText(this.getStatus(available, processing, pluginId));

    if (progress > -1) {
      this.progressContainer.setVisible(true);
      this.progressContainer.progressbar.updateProgress(progress, Math.round(100 * progress) + '%');
    } else {
      this.progressContainer.setVisible(false);
      this.progressContainer.progressbar.updateProgress(0);
    }

    this.editButton.setDisabled(processing);
    this.executeButton.setVisible(!processing);
    this.cancelButton.setVisible(processing);
  },

  getStatus: function (available, processing, pluginId) {
    if (available) {
      return processing ? PW.t('running') : PW.t('stopped');
    } else {
      var info = PW.t('Zadanie_nie_jest_dostepne');
      if (pluginId) {
        info += ' (' + PW.t('plugin') + ': <b>' + pluginId + '</b>)';
      }
      return info;
    }
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.TaskControlToolbar.superclass.initComponent.call(this);

    this.addEvents(
      /**
       * Zażądano dostępu do historii
       * @param toolbar toolbar zadania
       * @param showLogs czy po pokazaniu okna mają być automatycznie widoczne logi
       */
      'history',

      /**
       * Żądanie wykonania zadania
       * @param toolbar toolbar zadania
       */
      'execute',

      /**
       * Zażądano anulowania zadania
       * @param toolbar toolbar zadania
       */
      'cancel',

      /**
       * Zażądano usunięcia zadania
       * @param toolbar toolbar zadania
       */
      'remove',

      /**
       * Zadanie zmieniło wartość 'active'
       */
      'activechange',
      'beforeparametersrender'
    );
  },

  requestActiveToggle: function (active) {
    Ext.Ajax.request({
      url: 'api/scheduledtasks/active',
      method: 'PUT',
      jsonData: { id: this.taskId, active: active },
      scope: this,
      callback: function (options, success, response) {
        try {
          if (success) {
            var responseDecoded = Ext.decode(response.responseText);
            if (responseDecoded.success === true) {
              this.onActiveToggleSuccess(active);
            } else {
              this.onActiveToggleFailure();
            }
          } else {
            this.onActiveToggleFailure();
          }
        } catch (error) {
          this.onActiveToggleFailure();
        }
      },
    });
  },

  requestCategryChange: function (categoryId, category, button) {
    button.disable();
    Ext.Ajax.request({
      url: 'com.plusmpm.servlet.scheduledTasks.ScheduledTasksCategoryServlet.customServlet',
      method: 'POST',
      params: { task: this.taskId, categoryId: categoryId, category: category },
      scope: this,
      success: function (response, opts) {
        Ext.StoreMgr.get('tasksCategoryStore').load({ callback: this.updateCategoryStore });
        button.enable();
      },
      failure: function (response, opts) {
        button.enable();
      },
    });
  },
}); // end of Ext.extend
Ext.reg('st_taskcontrol', Ext.ux.plusmpm.scheduledtasks.TaskControlToolbar);
