jQuery.extend(PW, {
    /**
     * Provides support for internationalization (i18n).
     *
     * @class PW.I18N
     */
    I18N: {
        cached: {
            browser: {},
            server: {}
        },

        bundle: function(translatorName, scope) {
            return PW.I18N.bundleForLanguage(translatorName, scope, Suncode.getUserLocale());
        },

        bundleForLanguage: function(translatorName, scope, language) {
            scope = scope || 'browser';

            var bundle = PW.I18N.cached[scope][language];
            var found = Ext.isObject(bundle);
            if (found) {
                bundle = bundle[translatorName];
                found = Ext.isObject(bundle);
            }
            if (!found) {
                jQuery.ajax({
                    url: Suncode.getAbsolutePath('/api/translation/bundle/cached/' + encodeURIComponent(translatorName)),
                    method: 'GET',
                    async: false,
                    timeout: 30000,
                    data: {
                        lang: language,
                        scope: scope.toUpperCase()
                    },
                    success: function(response) {
                        bundle = response;
                        if (!PW.I18N.cached[scope][language]) {
                            PW.I18N.cached[scope][language] = {};
                        }
                        PW.I18N.cached[scope][language][translatorName] = bundle;
                    }
                });
            }

            return bundle || {};
        },

        /**
         * Returns translated message for given key and translator name.
         *
         * @example
         *  PW.I18N.t('plusworkflow', 'user.desc', 'Some', 'Name')
         *
         * @method t
         * @param {String} translator translator name
         * @param {String} key message key
         * @param {...Object} [args] additional message arguments (resolved as placeholders i.e. {0})
         * @return Translated message or supplied key if there is no translation available
         */
        t: function(translator, key) {
            Array.prototype.splice.call(arguments, 2, 0, Suncode.getUserLocale());
            return PW.I18N.tForLanguage.apply(PW.I18N, arguments);
        },

        /**
         * Returns translated message for given key, translator name and language.
         *
         * @example
         *  PW.I18N.t('plusworkflow', 'user.desc', 'pl', 'Some', 'Name')
         *
         * @method t
         * @param {String} translator translator name
         * @param {String} key message key
         * @param {String} language language
         * @param {...Object} [args] additional message arguments (resolved as placeholders i.e. {0})
         * @return Translated message or supplied key if there is no translation available
         */
        tForLanguage: function(translator, key, language) {
            var message = PW.I18N.bundleForLanguage(translator, null, language)[key];
            if (Ext.isEmpty(message)) {
                PW.Logger.error('No text for key: ' + key);
                message = key;
            }

            if (arguments.length > 3) {
                var args = Array.prototype.slice.call(arguments, 3);
                message = PW.I18N._format(message, args);
            }

            return message;
        },

        scopeT: function(translator, scope, key) {
            Array.prototype.splice.call(arguments, 3, 0, Suncode.getUserLocale());
            return PW.I18N.scopeTForLanguage.apply(PW.I18N, arguments);
        },

        scopeTForLanguage: function(translator, scope, key, language) {
            var message = PW.I18N.bundleForLanguage(translator, scope, language)[key];
            if (Ext.isEmpty(message)) {
                PW.Logger.error('No text for key: ' + key);
                message = key;
            }

            if (arguments.length > 4) {
                var args = Array.prototype.slice.call(arguments, 4);
                message = PW.I18N._format(message, args);
            }

            return message;
        },

        getTranslators: function(scope) {
            return PW.I18N.getTranslatorsForLanguage(scope, Suncode.getUserLocale());
        },

        getTranslatorsForLanguage: function(scope, language) {
            scope = scope || 'browser';
            return PW.I18N.cached[scope][language];
        },

        /**
         * Creates shorthand function for translating message using single translator. For example instead of
         *
         *  PW.I18N.t('plusworkflow', 'user.desc', 'Some', 'Name')
         *
         * you can use create shorthand function once
         *
         *  PW.t = PW.I18N.createT('plusworkflow');
         *
         * and then use it like
         *
         *  PW.t('user.desc', 'Jan', 'Kowalski' );
         *
         * @method createT
         * @param {String} translator translator name
         */
        createT: function(translator, scope){
            return function() {
                var args = Array.prototype.slice.call(arguments);
                if(scope) {
                	args.unshift(scope);
                	args.unshift(translator);
                	return PW.I18N.scopeT.apply(PW.I18N, args);
                }
                else {
                	args.unshift(translator);
                	return PW.I18N.t.apply(PW.I18N, args);
                }
            }
        },

        _format: function(text, args) {
            args.forEach(function(item, index) {
                text = text.replace('{'+index+'}', item);
            });
            return text;
        }
    }
});

//Alias dla tłumaczeń 'plusworkflow'
PW.t = PW.I18N.createT('plusworkflow');