Ext.namespace( 'Ext.ux.plusmpm.scheduledtasks' );

/**
 * Okno podglądu historii i logów
 */
Ext.ux.plusmpm.scheduledtasks.HistoryWindow = function( config ) {

    var pagesize = 50;

    var store = new Ext.data.JsonStore( {
        autoDestroy: true,
        storeId: 'taskHistory',
        remoteSort: true,
        autoLoad: {
            params: {
                start: 0,
                limit: pagesize
            }
        },
        proxy: new Ext.data.HttpProxy( {
            method: 'GET',
            url: 'api/scheduledtasks/history/by/task'
        } ),
        baseParams: {
            taskId: config.taskId
        },
        listeners: {
            load: this.onHistoryStoreLoad,
            scope: this
        },
        root: 'data',
        totalProperty: 'total',
        idProperty: 'id',
        fields: [ 'id', {
            name: 'startDate',
            type: 'date',
            dateFormat: 'Y-m-d H:i:s'
        }, {
            name: 'finishDate',
            type: 'date',
            dateFormat: 'Y-m-d H:i:s'
        }, {
            name: 'success',
            type: 'boolean',
            dateFormat: 'Y-m-d H:i:s'
        },
            'result',
            'resultFiles'
        ]
    } );

    var colModel = new Ext.grid.ColumnModel( {
        columns: [ {
            header: PW.t('startDate'),
            width: 155,
            dataIndex: 'startDate',
            xtype: 'datecolumn',
            format: 'Y-m-d H:i:s'
        }, {
            header: PW.t('endDate'),
            width: 155,
            dataIndex: 'finishDate',
            xtype: 'datecolumn',
            format: 'Y-m-d H:i:s'
        }, {
            header: PW.t('success'),
            dataIndex: 'success',
            scope: this,
            width: 100,
            align: 'center',
            renderer: function( value, p, record ) {

                var trueStyle = "color: green;";
                var falseStyle = "color: red;";
                if ( value ) {
                    return String.format( '<span style=\"' + trueStyle + '";>{0}<span>', PW.t('yes') );
                }
                else
                    return String.format( '<span style=\"' + falseStyle + '";>{0}<span>', PW.t('no') );
            }
        }, {
            id: 'resultColumn',
            header: PW.t('result'),
            dataIndex: 'result',
            renderer: function( value, p, record ) {
                p.attr = "style=\"white-space:normal;\"";
                return value;
            }
        }, {
            id: 'resultFilesColumn',
            header: PW.t('resultFiles'),
            dataIndex: 'resultFiles',
            width: 210,
            renderer: function(value, metadata) {
                if (value.length !== 0) {
                    metadata.css = 'link-content';
                    return value.map(
                        file => `<a href="api/customfiles?id=${file.id}">${file.fileName}</a>`
                    ).join('<br>');
                }
                return PW.t('Brak');
            }
        }]
    } );

    var pagingToolbar = new Ext.PagingToolbar( {
        store: store,
        displayInfo: true,
        pageSize: pagesize,
        prependButtons: true,
        items: [ {
            xtype: 'button',
            text: PW.t('logsViewer'),
            iconCls: 'dvnt-icon-document',
            scope: this,
            enableToggle: true,
            pressed: false,
            toggleHandler: this.onShowLogsToggle,
            ref: 'showLogsBtn'
        },{ xtype: 'spacer', width: 150 } ]
    } );

    config = Ext.apply( {
        title: config.taskName + ' - ' + PW.t('executionHistory'),
        modal: true,
        width: 870,
        height: 550,
        layout: 'border',
        showLogs: config.showLogs,
        items: [ {
            bbar: pagingToolbar,
            region: 'center',
            xtype: 'grid',
            padding: 10,
            border: false,
            store: store,
            colModel: colModel,
            autoExpandColumn: 'resultColumn',
            viewConfig: {
                forceFit: false,
                emptyText: this.emptyGridText
            },
            sm: new Ext.grid.RowSelectionModel( {
                singleSelect: true,
                listeners: {
                    scope: this,
                    rowselect: this.onRowSelect
                }
            } ),
            ref: 'historyGrid'
        }, {
            region: 'south',
            xtype: 'panel',
            split: true,
            height: 200,
            hidden: true,
            border: false,
            ref: 'logsViewer',
            layout: 'fit',
            bbar: {
                xtype: 'paging',
                displayInfo: true,
                prependButtons: true,
                pageSize: this.logsChunkSize,
                store: {
                    xtype: 'st_filestore',
                    baseParams: {
                        task: config.taskId
                    },
                    listeners: {
                        scope: this,
                        load: this.onLogsLoadSuccess,
                        exception: this.onLogsLoadFailure,
                        beforeload: this.onLogsBeforeLoad
                    }
                },
                listeners: {
                    scope: this,
                    beforechange: function(paging, opts){
                        opts.log = this.historyGrid.getSelectionModel().getSelected().data.id;
                    }
                }
            },
            items: [ {
                xtype: 'textarea',
                readOnly: true,
                style: 'border: 0'
            } ]
        } ]
    }, config );

    Ext.ux.plusmpm.scheduledtasks.HistoryWindow.superclass.constructor.call( this, config );

}; // end of Ext.ux.plusmpm.scheduledtasks.HistoryWindow constructor

Ext.extend( Ext.ux.plusmpm.scheduledtasks.HistoryWindow, Ext.Window, {

    // liczba wczytywanych linii logu
    logsChunkSize: 1000,

    onShowLogsToggle: function( btn, pressed ) {
        this.viewLogs = pressed;

        if ( pressed ) {
            var selected = this.historyGrid.getSelectionModel().getSelected();
            this.onLogsShow( selected );
        }
        else {
            this.logsViewer.setVisible( false );
            this.doLayout();
        }
    },

    onRowSelect: function( sm, index, record ) {
        if ( this.viewLogs ) {
            this.onLogsShow( record );
        }
    },

    /**
     * Wczytuje logi z serwera dla aktualnie zaznaczonego rekordu,
     */
    onLogsShow: function( selected ) {
        if ( selected ) {
            this.logsViewer.setVisible( true );
            this.loadLogs( selected.data.id );
            this.logsViewer.items.first().setValue( '' );
            this.doLayout();
        }
    },

    onLogsLoadSuccess: function( store, logs ) {
        this.viewerMask.hide();
        this.logsViewer.items.first().setValue( logs );
    },

    onLogsLoadFailure: function() {
        this.viewerMask.hide();
        this.logsViewer.items.first().setValue( PW.t('noAvailableTasks') );
        this.logsViewer.doLayout();
    },

    onLogsBeforeLoad: function() {
        this.viewerMask.show();
    },

    onViewerAfterRender: function( viewer ) {
        this.viewerMask = new Ext.LoadMask( this.logsViewer.body );
    },

    onHistoryStoreLoad: function() {
        var sm = this.historyGrid.getSelectionModel();
        if(sm.getCount() == 0){
            sm.selectFirstRow();
        }
        
        if ( this.showLogs ) {
            this.historyGrid.getBottomToolbar().showLogsBtn.toggle( true );
        }
    },

    initComponent: function() {
        Ext.ux.plusmpm.scheduledtasks.HistoryWindow.superclass.initComponent.call( this );
        this.logsViewer.on( 'afterrender', this.onViewerAfterRender, this );
        this.on( 'destroy', this.onDestroy, this );
    },

    onDestroy: function( comp ) {
        Ext.destroy( this.viewerMask );
    },

    loadLogs: function( historyId ) {
        this.logsViewer.getBottomToolbar().store.load( {
            params: {
                start: 0,
                limit: this.logsChunkSize,
                log: historyId
            }
        } );
    }
} ); // end of Ext.extend

/**
 * Atrapa store'a aby mógł współpracować z pagingToolbar przy ładowaniu pliku
 * chunkami
 */
Ext.ux.plusmpm.scheduledtasks.FileChunkStore = Ext.extend( Ext.util.Observable, {

    paramNames: undefined,

    defaultParamNames: {
        start: 'start',
        limit: 'limit'
    },

    getCount: function() {
        return this.lines || 0;
    },

    getTotalCount: function() {
        return this.totalLength || 0;
    },

    load: function( options ) {
        options = Ext.apply( {}, options );
        Ext.applyIf( options.params, this.baseParams );

        this.fireEvent( 'beforeload', this, options );

        Ext.Ajax.request( {
            url: 'com.plusmpm.servlet.scheduledTasks.ScheduledTasksLogsServlet.customServlet',
            method: 'GET',
            params: options.params,
            scope: this,
            success: function( response, opts ) {
                var logFile = Ext.decode( response.responseText );
                this.data = logFile.content;
                this.totalLength = logFile.totalLines;
                this.lines = logFile.lines;
                this.fireEvent( 'load', this, this.data, options );
            },
            failure: function( response, opts ) {
                this.fireEvent( 'exception' );
            }
        } );
    },

    constructor: function( config ) {
        Ext.apply( this, config );

        this.baseParams = Ext.isObject( this.baseParams ) ? this.baseParams : {};
        this.paramNames = Ext.applyIf( this.paramNames || {}, this.defaultParamNames );

        this.addEvents(
        /**
         * @event exception
         *        <p>
         *        Fires if an exception occurs in the Proxy during a remote
         *        request. This event is relayed through the corresponding
         *        {@link Ext.data.DataProxy}. See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
         *        for additional details.
         * @param {misc}
         *            misc See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
         *            for description.
         */
        'exception',
        /**
         * @event beforeload Fires before a request is made for a new data
         *        object. If the beforeload handler returns <tt>false</tt> the
         *        {@link #load} action will be canceled.
         * @param {Store}
         *            this
         * @param {Object}
         *            options The loading options that were specified (see
         *            {@link #load} for details)
         */
        'beforeload',
        /**
         * @event load Fires after a new set of Records has been loaded.
         * @param {Store}
         *            this
         * @param {Ext.data.Record[]}
         *            records The Records that were loaded
         * @param {Object}
         *            options The loading options that were specified (see
         *            {@link #load} for details)
         */
        'load' );

        Ext.ux.plusmpm.scheduledtasks.FileChunkStore.superclass.constructor.call( this );
    }
} ); // end of Ext.extend
Ext.reg( 'st_filestore', Ext.ux.plusmpm.scheduledtasks.FileChunkStore );
