package com.suncode.autoupdate.tomcat;

import lombok.SneakyThrows;
import org.apache.catalina.Container;
import org.apache.catalina.Context;
import org.apache.catalina.Globals;
import org.apache.catalina.Host;
import org.apache.catalina.core.ApplicationContext;
import org.apache.catalina.core.ApplicationContextFacade;
import org.apache.catalina.core.StandardContext;
import org.apache.catalina.util.ServerInfo;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import javax.servlet.ServletContext;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.CodeSource;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

public class Tomcat7
    implements Impl
{
    @Override
    public boolean active()
    {
        return ServerInfo.getServerNumber().startsWith( "7" );
    }

    @SneakyThrows
    public File getRoot( Context context )
    {
        StandardContext ctx = (StandardContext) context;

        Method getBasePath = StandardContext.class.getDeclaredMethod( "getBasePath" );
        getBasePath.setAccessible( true );
        return new File( (String) getBasePath.invoke( ctx ) );
    }

    @Override
    @SneakyThrows
    public Context getContext( ServletContext servletContext )
    {
        Field facadeContextField = ApplicationContextFacade.class.getDeclaredField( "context" );
        Field contextField = ApplicationContext.class.getDeclaredField( "context" );
        facadeContextField.setAccessible( true );
        contextField.setAccessible( true );
        return (Context) contextField.get( facadeContextField.get( servletContext ) );
    }

    @Override
    public boolean isHookRegisterd( ServletContext context )
    {
        StandardContext standardContext = (StandardContext) getContext( context );
        File contextFile = locateDefaultContextFile( standardContext );

        Document document = readXml( contextFile );
        return isAlreadyRegistered( document );
    }

    @Override
    @SneakyThrows
    public void registerHook( ServletContext context, InputStream bootstrapJar )
    {
        StandardContext standardContext = (StandardContext) getContext( context );
        File contextFile = locateDefaultContextFile( standardContext );

        copyToLib( bootstrapJar );

        Document document = readXml( contextFile );
        if ( isAlreadyRegistered( document ) )
        {
            return;
        }

        Element listenerEl = document.createElement( "Listener" );
        listenerEl.setAttribute( "className", Bootstrap.class.getName() );
        document.getDocumentElement().appendChild( listenerEl );

        Host host = getHost( standardContext );
        host.setAutoDeploy( false );
        saveXml( document, contextFile );
    }

    @SneakyThrows
    private File locateDefaultContextFile( StandardContext standardContext )
    {
        File base = new File( System.getProperty( Globals.CATALINA_BASE_PROP ) );
        File contextFile = null;
        if ( standardContext.getDefaultContextXml() != null )
        {
            contextFile = new File( standardContext.getDefaultContextXml() );
            if ( !contextFile.isAbsolute() )
            {
                contextFile = new File( base, standardContext.getDefaultContextXml() );
            }
        }
        else
        {
            contextFile = new File( base, "conf/context.xml" );
        }
        if ( !contextFile.exists() )
        {
            // TODO: this may correct option without checking default context xml
            URL configFile = standardContext.getConfigFile();
            if(configFile != null) {
                contextFile = new File(configFile.toURI());
            }
            else {
                throw new RuntimeException("Context file not found: " + contextFile.getAbsolutePath());
            }
        }
        return contextFile;
    }

    private boolean isAlreadyRegistered( Document document )
    {
        NodeList listeners = document.getDocumentElement().getElementsByTagName( "Listener" );
        for ( int i = 0; i < listeners.getLength(); i++ )
        {
            Element listener = (Element) listeners.item( i );
            if ( listener.getAttribute( "className" ).equals( Bootstrap.class.getName() ) )
            {
                return true;
            }
        }
        return false;
    }

    @SneakyThrows
    private Document readXml( File source )
    {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder documentBuilder = factory.newDocumentBuilder();
        try (InputStream inputStream = new FileInputStream( source ))
        {
            return documentBuilder.parse( inputStream );
        }
    }

    private void saveXml( Document document, File output )
        throws TransformerFactoryConfigurationError, IOException, TransformerException
    {
        Transformer transformer = TransformerFactory.newInstance().newTransformer();
        try (OutputStream outputStream = new FileOutputStream( output ))
        {
            Source input = new DOMSource( document );
            Result result = new StreamResult( outputStream );
            transformer.transform( input, result );
        }
    }

    private Host getHost( Context context )
    {
        Container parent = context.getParent();
        while ( parent != null )
        {
            if ( parent instanceof Host )
            {
                return (Host) parent;
            }
            parent = context.getParent();
        }
        throw new IllegalStateException( "Could not access Host from " + context );
    }

    @SneakyThrows
    private Path copyToLib( InputStream inputStream )
    {
        CodeSource libSource = StandardContext.class.getProtectionDomain().getCodeSource();
        if ( libSource == null )
        {
            throw new IllegalStateException( "Cant locate tomcat lib directory" );
        }

        Path libDirectory = Paths.get( libSource.getLocation().toURI() ).getParent();
        Path target = libDirectory.resolve( "autoupdate-tomcat-hook.jar" );

        try
        {
            Files.copy( inputStream, target, REPLACE_EXISTING );
        }
        catch ( IOException e )
        {
            if ( !Files.exists( target ) )
            {
                throw e;
            }
        }
        return target;
    }
}
