package com.suncode.autoupdate.server.event;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Path;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.convert.converter.Converter;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.fasterxml.jackson.databind.ObjectMapper;

@RestController
public class EventController
{
    @Autowired
    private Events events;

    @Autowired
    private EventRepository eventRepository;

    @Autowired
    private ObjectMapper objectMapper;

    @RequestMapping( value = "/events/update/success", method = RequestMethod.POST )
    public void updateSuccess( @RequestParam String patchId, @RequestParam String clientEnv, @RequestParam String data )
        throws Exception
    {
        events.updateSuccess( patchId, clientEnv, data );
    }

    @RequestMapping( value = "/events/update/error", method = RequestMethod.POST )
    public void updateError( @RequestParam String patchId, @RequestParam String clientEnv, @RequestParam String data )
        throws Exception
    {
        events.updateError( patchId, clientEnv, data );
    }

    @RequestMapping( value = "/events/rollback/success", method = RequestMethod.POST )
    public void rollbackSuccess( @RequestParam String patchId, @RequestParam String clientEnv,
                                 @RequestParam String data )
                                     throws Exception
    {
        events.rollbackSuccess( patchId, clientEnv, data );
    }

    @RequestMapping( value = "/events/rollback/error", method = RequestMethod.POST )
    public void rollbackError( @RequestParam String patchId, @RequestParam String clientEnv, @RequestParam String data )
        throws Exception
    {
        events.rollbackError( patchId, clientEnv, data );
    }

    @RequestMapping( value = "/events", method = RequestMethod.GET )
    public Page<EventDto> query( Pageable pageable, @RequestParam String filter )
        throws Exception
    {
        final EventsFilter filters = objectMapper.readValue( filter, EventsFilter.class );
        Specification<Event> spec = new Specification<Event>() {

            @Override
            public Predicate toPredicate( Root<Event> root, CriteriaQuery<?> query, CriteriaBuilder cb )
            {
                List<Predicate> predicates = new ArrayList<>();

                // eventType
                List<Predicate> types = new ArrayList<>();
                for ( EventType patchType : filters.getPatchType() )
                {
                    types.add( cb.equal( root.get( "type" ), patchType ) );
                }
                if ( !types.isEmpty() )
                {
                    predicates.add( cb.or( types.toArray( new Predicate[types.size()] ) ) );
                }

                // dateFrom
                if ( filters.getDateFrom() != null )
                {
                    predicates.add( cb.greaterThan( root.<Date> get( "date" ), filters.getDateFrom() ) );
                }

                // dateTo
                if ( filters.getDateTo() != null )
                {
                    predicates.add( cb.lessThan( root.<Date> get( "date" ), filters.getDateTo() ) );
                }

                // client
                if ( StringUtils.hasText( filters.getClient() ) )
                {
                    predicates
                        .add( cb.like( cb.lower( root.<String> get( "client" ) ),
                                       "%" + filters.getClient().toLowerCase() + "%" ) );
                }

                // project
                if ( StringUtils.hasText( filters.getProject() ) )
                {
                    Path<String> path = root.get( "patch").get( "channel").get( "id").get( "projectName" );
                    predicates
                        .add( cb.like( cb.lower( path ),
                                       "%" + filters.getProject().toLowerCase() + "%" ) );
                }

                // channel
                if ( StringUtils.hasText( filters.getChannel() ) )
                {
                    Path<String> path = root.get( "patch").get( "channel").get( "id").get( "name" );
                    predicates
                        .add( cb.like( cb.lower(path ),
                                       "%" + filters.getChannel().toLowerCase() + "%" ) );
                }

                return cb.and( predicates.toArray( new Predicate[predicates.size()] ) );
            }
        };
        return eventRepository.findAll( spec, pageable ).map( new Converter<Event, EventDto>() {
            @Override
            public EventDto convert( Event source )
            {
                return new EventDto( source );
            }
        } );
    }
}
