package com.suncode.autoupdate.server.store;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.autoupdate.server.client.api.ProjectId;
import com.suncode.autoupdate.server.store.wooapi.Product;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.Value;
import lombok.experimental.FieldDefaults;
import lombok.extern.slf4j.Slf4j;
import org.jooq.DSLContext;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import suncode.jooq.tables.StoreProducts;

import java.sql.Timestamp;
import java.time.Clock;
import java.util.Collection;
import java.util.List;

import static com.suncode.autoupdate.server.client.api.ProjectId.projectId;
import static java.util.stream.Collectors.toList;
import static lombok.AccessLevel.PRIVATE;
import static org.jooq.impl.DSL.select;
import static suncode.jooq.tables.StoreMapping.STORE_MAPPING;
import static suncode.jooq.tables.StoreProducts.STORE_PRODUCTS;

@Slf4j
@Component
@Transactional
@RequiredArgsConstructor
@FieldDefaults(level = PRIVATE, makeFinal = true)
class ProductsStorage {
    DSLContext jooq;
    ObjectMapper mapper;
    Clock clock;

    @CacheEvict(cacheNames = {
            "store.products.notMapped"
    }, allEntries = true)
    public void save(Collection<Product> products) {
        jooq.batch(products.stream()
                .map(product -> jooq.insertInto(STORE_PRODUCTS)
                        .set(STORE_PRODUCTS.ID, product.getId().get())
                        .set(STORE_PRODUCTS.PRODUCT, serialize(product))
                        .set(STORE_PRODUCTS.CREATED, Timestamp.from(clock.instant()))
                        .onDuplicateKeyUpdate()
                        .set(STORE_PRODUCTS.PRODUCT, serialize(product))
                        .set(STORE_PRODUCTS.UPDATED, Timestamp.from(clock.instant()))
                )
                .collect(toList())
        ).execute();
    }

    @CacheEvict(cacheNames = {
            "store.products.mapped",
            "store.products.notMapped"
    }, allEntries = true)
    public void addMapping(ProjectId projectId, Product.Id productPl, Product.Id productEn) {
        jooq.insertInto(STORE_MAPPING)
                .set(STORE_MAPPING.PROJECT_ID, projectId.asString())
                .set(STORE_MAPPING.PRODUCT_EN, productEn.get())
                .set(STORE_MAPPING.PRODUCT_PL, productPl.get())
                .execute();
        log.info("Added mapping for projectId={} pl={} en={}", projectId, productPl, productEn);
    }

    @CacheEvict(cacheNames = {
            "store.products.mapped",
            "store.products.notMapped"
    }, allEntries = true)
    public void deleteMapping(ProjectId projectId) {
        jooq.deleteFrom(STORE_MAPPING)
                .where(STORE_MAPPING.PROJECT_ID.eq(projectId.asString()))
                .execute();
        log.info("Deleted mapping for projectId={}", projectId);

    }

    @Cacheable("store.products.mapped")
    public List<Mapped> getMappedProducts() {
        StoreProducts pl = STORE_PRODUCTS.as("product_pl");
        StoreProducts en = STORE_PRODUCTS.as("product_en");

        return jooq.selectFrom(
                STORE_MAPPING
                        .join(pl).on(pl.ID.eq(STORE_MAPPING.PRODUCT_PL))
                        .join(en).on(en.ID.eq(STORE_MAPPING.PRODUCT_EN)))
                .fetch()
                .map(record -> new Mapped(
                        projectId(record.get(STORE_MAPPING.PROJECT_ID)),
                        deserialize(record.get(pl.PRODUCT, String.class)),
                        deserialize(record.get(en.PRODUCT, String.class))
                ));
    }

    @Cacheable("store.products.notMapped")
    public List<Product> getNotMappedProducts() {
        return jooq.selectFrom(STORE_PRODUCTS)
                .where(STORE_PRODUCTS.ID.notIn(
                        select(STORE_MAPPING.PRODUCT_PL).from(STORE_MAPPING)
                ))
                .and(STORE_PRODUCTS.ID.notIn(
                        select(STORE_MAPPING.PRODUCT_EN).from(STORE_MAPPING)
                ))
                .fetch(record -> deserialize(record.component2()));
    }

    @SneakyThrows
    private String serialize(Product product) {
        return mapper.writeValueAsString(product);
    }

    @SneakyThrows
    private Product deserialize(String raw) {
        return mapper.readValue(raw, Product.class);
    }


    @Value
    static class Mapped {
        ProjectId projectId;
        Product productPl;
        Product productEn;
    }
}
