package com.suncode.autoupdate.server.project;

import com.google.common.collect.Iterables;
import com.suncode.autoupdate.server.client.Client;
import com.suncode.autoupdate.server.client.ClientRepository;
import com.suncode.autoupdate.server.event.EventRepository;
import com.suncode.autoupdate.server.patch.PatchFormat;
import com.suncode.autoupdate.server.project.dto.ProjectDto;
import com.suncode.autoupdate.server.security.HasAccess;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.annotation.Secured;
import org.springframework.security.web.servletapi.SecurityContextHolderAwareRequestWrapper;
import org.springframework.web.bind.annotation.*;

import javax.transaction.Transactional;
import java.util.Collections;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.suncode.autoupdate.server.security.HasAccess.checkAccessToProject;

@RestController
@RequestMapping( "/projects" )
public class ProjectController
{
    @Autowired
    private ProjectRepository projectRepo;

    @Autowired
    private ClientRepository clientRepo;

    @Autowired
    private EventRepository eventRepo;

    @RequestMapping( method = RequestMethod.GET )
    public List<ProjectDto> get( SecurityContextHolderAwareRequestWrapper request )
    {
        return ProjectDto.from( Iterables.filter( projectRepo.findAll(), HasAccess.withAccessPredicate(request)::test) );
    }

    @RequestMapping( value = "/{name:.+}", method = RequestMethod.GET )
    public ProjectDto get( @PathVariable String name, SecurityContextHolderAwareRequestWrapper request )
    {
        Project project = projectRepo.findOne(name);
        checkAccessToProject(project, request);
        return new ProjectDto(project);
    }

    @Secured("ROLE_ADMIN")
    @RequestMapping( method = RequestMethod.POST )
    public ProjectDto create( @RequestParam String name, @RequestParam String type )
    {
        Project project = new Project( name, PatchFormat.valueOf( type ) );
        projectRepo.save( project );
        return new ProjectDto( project );
    }


    @Secured("ROLE_ADMIN")
    @Transactional
    @RequestMapping( value = "/{name:.+}", method = RequestMethod.DELETE )
    public void delete( @PathVariable String name )
    {
        eventRepo.deleteByProject( name );
        projectRepo.delete( name );
    }


    @Secured("ROLE_ADMIN")
    @RequestMapping( value = "/{name:.+}/permittedClients", method = RequestMethod.PUT )
    public void updatePermittedClients( @PathVariable String name, @RequestBody(required=false) List<String> clients )
    {
        Project project = projectRepo.findOne( name );
        List<Client> permittedClients = clients != null
                        ? newArrayList(clientRepo.findAll( clients ))
                        : Collections.<Client>emptyList();
        project.permitClients( permittedClients );
        projectRepo.save( project );
    }
}
