package com.suncode.autoupdate.server.patch.types;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.suncode.autoupdate.patch.PatchMeta;
import com.suncode.autoupdate.server.patch.PatchHandler;
import com.suncode.plugin.framework.Reference;
import com.suncode.plugin.framework.Version;
import com.suncode.plugin.framework.config.PluginDescriptor;
import com.suncode.plugin.framework.config.PluginDescriptorReader;
import lombok.val;
import org.apache.commons.io.IOUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;
import java.util.jar.JarFile;

import static com.suncode.autoupdate.patch.plugin.PluginPatchProperties.*;
import static java.util.stream.Collectors.toSet;
import static org.osgi.framework.Version.parseVersion;

/**
 * Plugin patch.
 */
public class PluginPatchFormat
        implements PatchHandler {
    @Override
    public boolean transforms(File source) {
        try {
            JarFile jarFile = new JarFile(source);
            jarFile.close();
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public Map<String, String> transform(PatchMeta patch, File source, OutputStream out)
            throws IOException {
        Version version = Version.parse(patch.getToVersion());

        try (FileInputStream in = new FileInputStream(source)) {
            IOUtils.copy(in, out);
        }

        ObjectMapper mapper = new ObjectMapper();
        PluginDescriptor descriptor = new PluginDescriptorReader().readDescriptor(source);
        Reference ref = Reference.of(descriptor.getKey(), version);

        val properties = ImmutableMap.<String, String>builder();

        properties.put(REQUIREMENTS, serializeRequirements(mapper, descriptor));
        descriptor.getDetails().getChangelog()
                .ifPresent(uri -> properties.put(CHANGELOG, uri.build(ref).toString()));
        descriptor.getDetails().getDocumentation()
                .ifPresent(uri -> properties.put(DOCUMENTATION, uri.build(ref).toString()));

        return properties.build();
    }

    private static String serializeRequirements(ObjectMapper mapper, PluginDescriptor descriptor) throws JsonProcessingException {
        return mapper.writeValueAsString(
                descriptor.getDetails().getRequirements()
                        .stream()
                        .map(requirement -> new PluginRequirement(
                                requirement.getId(),
                                requirement.getVersion().toString(),
                                requirement.isMandatory()))
                        .collect(toSet()));
    }

    @Override
    public int compare(String version1, String version2) {
        return parseVersion(version1).compareTo(parseVersion(version2));
    }
}
