package com.suncode.autoupdate.server.patch.storage;

import com.google.common.hash.Hashing;
import com.google.common.hash.HashingOutputStream;
import com.suncode.autoupdate.server.channel.Channel.ChannelId;
import com.suncode.autoupdate.server.patch.Patch;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import java.io.*;

/**
 * {@link PatchStorage} implementation that stores patches on local file system.
 * 
 * @author Cezary Kozar 26 kwi 2016
 */
@Slf4j
@Component
public class FileSystemPatchStorage
    implements PatchStorage
{
    private final File root;

    public FileSystemPatchStorage()
        throws IOException
    {
        root = new File( "data/patches/" );
        FileUtils.forceMkdir( root );
    }

    @Override
    public void store( Patch patch, File patchFile )
        throws IOException
    {
        try (InputStream in = new FileInputStream( patchFile ))
        {
            store( patch, in );
        }
    }

    @Override
    public void store( Patch patch, final InputStream patchInput )
        throws IOException
    {
        store( patch, new StorageOutput() {

            @Override
            public void store( OutputStream out )
                throws IOException
            {
                IOUtils.copy( patchInput, out );
            }
        } );
    }

    @Override
    public void store( Patch patch, StorageOutput output )
        throws IOException
    {
        File destination = storageFile( patch );
        FileUtils.forceMkdir( destination.getParentFile() );

        try (HashingOutputStream out = new HashingOutputStream( Hashing.md5(), new FileOutputStream( destination ) ))
        {
            output.store( out );
            patch.setChecksum( out.hash().toString() );
        }
        log.info( "Patch {} stored in {}", patch.getId(), destination.getAbsolutePath() );
    }

    @Override
    public Resource read( Patch patch )
    {
        File archive = storageFile( patch );
        if ( !archive.exists() )
        {
            throw new IllegalStateException( "Patch not found in [" + archive.getAbsolutePath() + "]!" );
        }
        return new FileSystemResource( archive );
    }

    @Override
    public void purge( ChannelId channel ) throws IOException
    {
        FileUtils.cleanDirectory( channelStorageFile( channel ) );
    }

    private File storageFile( Patch patch )
    {
        return new File( channelStorageFile( patch.getChannel().getId() ), patch.getId() + ".patch" );
    }

    private File channelStorageFile( ChannelId channel )
    {
        return new File( root, channel.getProjectName() + "/" + channel.getName() );
    }
}
