package com.suncode.autoupdate.server.client;

import static com.suncode.autoupdate.server.client.Client.Type.STANDARD;
import static com.suncode.autoupdate.server.client.Client.Type.TRIAL;
import static org.apache.commons.lang3.BooleanUtils.toBoolean;

import java.time.Instant;
import java.util.*;

import javax.persistence.*;

import lombok.Getter;
import org.hibernate.annotations.CreationTimestamp;
import org.springframework.util.Assert;

import com.suncode.autoupdate.server.project.Project;

@Entity
public class Client
{
    public enum Type {
        TRIAL,
        STANDARD;
    }

    @Id
    private String id;

    @Column( unique = true, nullable = false )
    private String name;

    @Column
    private String apiToken;

    @Column
    private Type type = STANDARD;

    @Column
    private Boolean compressPatches;

    @Getter
    @CreationTimestamp
    @Temporal(TemporalType.TIMESTAMP)
    @Column(nullable = false)
    private Date created;

    @ElementCollection( fetch = FetchType.EAGER )
    @CollectionTable( name = "client_env", joinColumns = @JoinColumn( name = "id" ))
    private List<Environment> environments = new ArrayList<>();

    @ManyToMany
    @JoinTable( name = "project_permitted_clients", joinColumns = @JoinColumn( name = "permitted_clients_id" ), inverseJoinColumns = @JoinColumn( name = "project_name" ) )
    private Set<Project> permittedProjects = new HashSet<>();

    Client()
    {
    }

    public Client( String id, String name )
    {
        Assert.hasText( id );
        Assert.hasText( name );
        this.id = id;
        this.name = name;
        this.created = new Date();
    }

    public static Client trial()
    {
        UUID id = UUID.randomUUID();
        Client client = new Client();
        client.id = id.toString();
        client.name = id.toString();
        client.type = TRIAL;

        client.generateApiToken();
        return client;
    }

    public String getId()
    {
        return id;
    }

    public String getName()
    {
        return name;
    }

    public void setName( String name )
    {
        this.name = name;
    }

    public Type getType() {
        return Optional.ofNullable(type).orElse(STANDARD);
    }

    public String getApiToken()
    {
        return apiToken;
    }

    public void generateApiToken()
    {
        this.apiToken = UUID.randomUUID().toString();
    }

    public boolean compressPatches()
    {
        return getCompressPatches();
    }

    public Boolean getCompressPatches()
    {
        return toBoolean( compressPatches );
    }

    public void setCompressPatches( Boolean compressPatches )
    {
        this.compressPatches = compressPatches;
    }

    public List<Environment> getEnvironments()
    {
        return environments;
    }
}
