package com.suncode.autoupdate.server.channel;

import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patch.plusworkflow.archive.merge.ArchiveMerger;
import com.suncode.autoupdate.server.channel.Channel.ChannelId;
import com.suncode.autoupdate.server.channel.dto.ChannelGraphDto;
import com.suncode.autoupdate.server.channel.dto.UpdateChannelDto;
import com.suncode.autoupdate.server.channel.graph.ChannelGraph;
import com.suncode.autoupdate.server.patch.Patch;
import com.suncode.autoupdate.server.patch.Version;
import com.suncode.autoupdate.server.patch.dto.PatchDto;
import com.suncode.autoupdate.server.patch.storage.PatchStorage;
import com.suncode.autoupdate.server.project.Project;
import com.suncode.autoupdate.server.project.ProjectRepository;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.web.bind.annotation.*;

import java.io.File;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;
import static org.springframework.http.ResponseEntity.ok;

@Slf4j
@RestController
@RequestMapping("/projects/{projectId:.+}/channels")
@Secured("ROLE_ADMIN")
public class ChannelController {
    @Autowired
    private Channels channels;

    @Autowired
    private ProjectRepository projectRepo;

    @Autowired
    private PatchStorage patchStorage;

    @Autowired
    private TempResourcesController tempResourcesController;

    @RequestMapping(value = "/{channelId}")
    public UpdateChannelDto getChannel(@PathVariable String projectId, @PathVariable String channelId) {
        UpdateChannel channel = channels.get(ChannelId.of(channelId, projectId));
        return new UpdateChannelDto(channel);
    }

    @RequestMapping(value = "")
    public List<UpdateChannelDto> getChannels(@PathVariable String projectId) {
        List<UpdateChannelDto> result = new ArrayList<>();

        Project project = projectRepo.findOne(projectId);
        for (Channel channel : project.getChannels()) {
            result.add(new UpdateChannelDto(channels.get(channel.getId())));
        }
        return result;
    }

    @RequestMapping(value = "/{channelId}", method = RequestMethod.POST)
    public UpdateChannelDto create(@PathVariable String projectId, @PathVariable String channelId) {
        UpdateChannel channel = channels.create(ChannelId.of(channelId, projectId));
        return new UpdateChannelDto(channel);
    }

    @RequestMapping(value = "/{channelId}", method = RequestMethod.DELETE)
    public void delete(@PathVariable String projectId, @PathVariable String channelId) {
        channels.remove(ChannelId.of(channelId, projectId));
    }

    @RequestMapping(value = "/{channelId}/graph", method = RequestMethod.GET)
    public ChannelGraphDto graph(@PathVariable String projectId, @PathVariable String channelId) {
        ChannelGraph graph = channels.get(ChannelId.of(channelId, projectId)).graph();
        return new ChannelGraphDto(graph);
    }

    @RequestMapping(value = "/{channelId}/patches", method = RequestMethod.GET)
    public List<PatchDto> list(@PathVariable String projectId, @PathVariable String channelId) {
        return channels.get(ChannelId.of(channelId, projectId)).getPatches().stream()
                .map(PatchDto::new)
                .collect(toList());
    }

    @SneakyThrows
    @GetMapping("/{channelId}/merge")
    public ResponseEntity merge(
            @PathVariable String projectId,
            @PathVariable String channelId,
            @RequestParam String from,
            @RequestParam String to
    ) {
        UpdateChannel channel = channels.get(ChannelId.of(channelId, projectId));

        List<Patch> patches = channel.graph().shortestPath(
                new Version(from),
                new Version(to)
        );

        if (patches.isEmpty()) {
            return ResponseEntity.badRequest()
                    .body("No connection exists between #" + from + " and #" + to);
        }

        log.info("Generating merged patch using {}", patches);

        ArchiveMerger merger = patches.stream()
                .map(patch -> patchStorage.read(patch))
                .map(resource -> archive(resource))
                .collect(Collectors.collectingAndThen(
                        toList(),
                        ArchiveMerger::new
                ));

        File output = File.createTempFile("patch-merged-", ".zip");
        output.deleteOnExit();

        log.info("Merging started");
        merger.merge(output);

        URI uri = tempResourcesController.serve(
                String.format("patch-#%s-#%s.patch", from, to),
                output
        );
        log.info("Merged patch generated under {}", uri);

        return ok(uri);
    }


    @SneakyThrows
    private Archive archive(Resource resource) {
        return new Archive(resource.getFile());
    }
}
