package com.suncode.autoupdate.server.channel;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Embeddable;
import javax.persistence.EmbeddedId;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.ManyToOne;
import javax.persistence.MapsId;
import javax.persistence.OneToMany;

import org.springframework.util.Assert;

import com.suncode.autoupdate.server.patch.PatchFormat;
import com.suncode.autoupdate.server.patch.Patch;
import com.suncode.autoupdate.server.project.Project;

/**
 * Channel entity.
 * 
 * @author Cezary Kozar 18 maj 2016
 */
@Entity
public class Channel
{
    @EmbeddedId
    private ChannelId id;

    @ManyToOne
    @MapsId( "projectName" )
    private Project project;

    @OneToMany( cascade = CascadeType.ALL, fetch = FetchType.EAGER, mappedBy = "channel" )
    private List<Patch> patches = new ArrayList<>();

    Channel()
    {
    }

    public Channel( String name, Project project )
    {
        Assert.hasText( name );
        Assert.notNull( project );

        this.id = new ChannelId( name, project.getName() );
        this.project = project;
    }

    public ChannelId getId()
    {
        return id;
    }

    public Project getProject()
    {
        return project;
    }

    public List<Patch> getPatches()
    {
        return patches;
    }

    public PatchFormat getPatchFormat()
    {
        return project.getPatchFormat();
    }

    @Embeddable
    @SuppressWarnings( "serial" )
    public static class ChannelId
        implements Serializable
    {
        private static final String NAME_REGEX = "[a-zA-Z][a-zA-Z0-9-_]{0,50}";

        private String name;

        private String projectName;

        ChannelId()
        {
        }

        public ChannelId( String name, String projectName )
        {
            Assert.hasText( name );
            Assert.hasText( projectName );
            Assert.isTrue( name.matches( NAME_REGEX ), "Channel name must match regex: " + NAME_REGEX );

            this.name = name.toLowerCase();
            this.projectName = projectName.toLowerCase();
        }

        public static ChannelId of( String name, String projectName )
        {
            return new ChannelId( name, projectName );
        }

        public String getName()
        {
            return name;
        }

        public String getProjectName()
        {
            return projectName;
        }

        @Override
        public boolean equals( Object obj )
        {
            if ( obj == this )
            {
                return true;
            }
            if ( obj == null )
            {
                return false;
            }
            if ( obj.getClass().equals( ChannelId.class ) )
            {
                ChannelId other = (ChannelId) obj;
                return this.name.equals( other.name ) && this.projectName.equals( other.projectName );
            }
            return super.equals( obj );
        }

        @Override
        public int hashCode()
        {
            final int prime = 31;
            int result = 1;
            result = prime * result + name.hashCode();
            result = prime * result + projectName.hashCode();
            return result;
        }

        @Override
        public String toString()
        {
            return "#" + name + "@" + projectName;
        }
    }
    
    @Override
    public String toString()
    {
        return id.toString();
    }
}
