package com.suncode.autoupdate.server;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.suncode.autoupdate.server.security.ApiTokenAuthenticationFilter;
import com.suncode.autoupdate.server.security.ApiTokenAuthenticationProvider;
import org.jooq.DSLContext;
import org.jooq.conf.Settings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.web.authentication.www.BasicAuthenticationFilter;
import org.springframework.web.servlet.i18n.AcceptHeaderLocaleResolver;

import java.time.Clock;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static com.fasterxml.jackson.databind.SerializationFeature.INDENT_OUTPUT;
import static org.jooq.conf.RenderNameCase.LOWER;
import static org.springframework.http.HttpMethod.POST;

@SpringBootApplication
@EnableCaching
@EnableWebSecurity
@EnableScheduling
@EnableGlobalMethodSecurity(securedEnabled = true, proxyTargetClass = true, prePostEnabled = true)
public class UpdateServer
        extends WebSecurityConfigurerAdapter {
    @Value("${user.username}")
    private String userName;

    @Value("${user.password}")
    private String userPassword;

    @Bean
    Clock clock() {
        return Clock.systemUTC();
    }

    @Bean
    Settings jooqSettings() {
        return new Settings()
                .withRenderNameCase(LOWER);
    }

    @Bean
    @Primary
    public ObjectMapper objectMapper(DSLContext jooq) {
        return new ObjectMapper()
                .registerModule(new Jdk8Module())
                .registerModule(new JavaTimeModule())
                .enable(INDENT_OUTPUT)
                .disable(FAIL_ON_UNKNOWN_PROPERTIES)
                .setSerializationInclusion(Include.ALWAYS);
    }

    @Bean
    @Override
    public AuthenticationManager authenticationManagerBean()
            throws Exception {
        return super.authenticationManagerBean();
    }

    @Bean
    public ApiTokenAuthenticationFilter apiTokenAuthenticationFilter()
            throws Exception {
        return new ApiTokenAuthenticationFilter(authenticationManagerBean());
    }

    @Bean
    public ApiTokenAuthenticationProvider apiTokenAuthenticationProvider() {
        return new ApiTokenAuthenticationProvider();
    }

    @Override
    public void configure(WebSecurity web)
            throws Exception {
        web.ignoring().antMatchers("/css/**", "/assets/**", "/fonts/**", "/img/**", "/bundle.js");
    }

    @Override
    protected void configure(HttpSecurity http)
            throws Exception {
        // @formatter:off
        http
                .sessionManagement()
                .sessionCreationPolicy(SessionCreationPolicy.STATELESS)
                .and()
                .csrf().disable()
                .authorizeRequests()
                .antMatchers(POST, "/trial").permitAll()
                .anyRequest().authenticated()
                .and()
                .addFilterBefore(apiTokenAuthenticationFilter(), BasicAuthenticationFilter.class)
                .httpBasic();

        // @formatter:on
    }

    @Autowired
    public void configureGlobal(AuthenticationManagerBuilder auth)
            throws Exception {
        auth
                .authenticationProvider(apiTokenAuthenticationProvider())
                .inMemoryAuthentication()
                .withUser(userName).password(userPassword).roles("ADMIN", "ACTUATOR");
    }

    @Bean
    public AcceptHeaderLocaleResolver localeResolver() {
        return new AcceptHeaderLocaleResolver();
    }

    public static void main(String[] args) {
        SpringApplication.run(UpdateServer.class, args);
    }
}
