package com.suncode.autoupdate.server.store;


import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.autoupdate.patch.plugin.PluginPatchProperties.PluginRequirement;
import com.suncode.autoupdate.server.channel.Channel.ChannelId;
import com.suncode.autoupdate.server.channel.Channels;
import com.suncode.autoupdate.server.channel.UpdateChannel;
import com.suncode.autoupdate.server.client.api.Plugin;
import com.suncode.autoupdate.server.client.api.Plugin.Unresolved;
import com.suncode.autoupdate.server.client.api.ProjectId;
import com.suncode.autoupdate.server.client.api.StoreCriteria;
import com.suncode.autoupdate.server.client.api.StoreCriteria.ClientCapabilities;
import com.suncode.autoupdate.server.client.api.StoreCriteria.ClientCapabilities.Component;
import com.suncode.autoupdate.server.patch.Patch;
import com.suncode.autoupdate.server.patch.PatchController;
import com.suncode.autoupdate.server.project.Project;
import com.suncode.autoupdate.server.project.ProjectRepository;
import com.suncode.autoupdate.server.store.wooapi.Product;
import com.suncode.plugin.framework.Reference;
import com.suncode.plugin.framework.Version;
import com.suncode.plugin.framework.requirements.Capabilities;
import com.suncode.plugin.framework.requirements.Capabilities.HostCapability;
import com.suncode.plugin.framework.requirements.Capabilities.PluginCapability;
import com.suncode.plugin.framework.requirements.Requirement;
import com.suncode.plugin.framework.requirements.Resolution;
import com.suncode.plugin.framework.requirements.Resolution.Visitor;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.Value;
import lombok.experimental.FieldDefaults;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.security.web.servletapi.SecurityContextHolderAwareRequestWrapper;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Stream;

import static com.suncode.autoupdate.patch.plugin.PluginPatchProperties.REQUIREMENTS;
import static com.suncode.autoupdate.server.client.api.Plugin.Unresolved.Error.MISSING;
import static com.suncode.autoupdate.server.client.api.Plugin.Unresolved.Error.VERSION_MISMATCH;
import static com.suncode.autoupdate.server.patch.PatchFormat.PLUGIN;
import static com.suncode.autoupdate.server.patch.Version.ANY;
import static com.suncode.autoupdate.server.security.HasAccess.hasAccessToProject;
import static com.suncode.autoupdate.server.security.HasAccess.withAccessPredicate;
import static com.suncode.autoupdate.server.util.Lists.map;
import static com.suncode.autoupdate.server.util.Predicates.compose;
import static com.suncode.autoupdate.server.util.Streams.parallelNotSupported;
import static com.suncode.plugin.framework.Plugin.PluginState.ACTIVE;
import static com.suncode.plugin.framework.Version.parse;
import static java.util.stream.Collectors.toList;
import static lombok.AccessLevel.PRIVATE;
import static org.springframework.hateoas.mvc.ControllerLinkBuilder.linkTo;
import static org.springframework.http.HttpStatus.FORBIDDEN;
import static org.springframework.http.ResponseEntity.*;
import static org.springframework.web.bind.annotation.RequestMethod.POST;

@Slf4j
@RestController
@RequestMapping("/store")
@RequiredArgsConstructor
@FieldDefaults(level = PRIVATE, makeFinal = true)
public class StoreResource {

    Channels channels;
    ProjectRepository projectRepository;
    ObjectMapper objectMapper;
    ProductsStorage productsStorage;
    Store store;

    @GetMapping("/products")
    public Collection<Product> products() {
        return productsStorage.getNotMappedProducts();
    }

    @GetMapping("/mappings")
    public Collection<ProductsStorage.Mapped> store() {
        return productsStorage.getMappedProducts();
    }

    @PostMapping("mappings")
    public void add(@RequestBody AddRequest request) {
        productsStorage.addMapping(
                request.getPlugin(),
                request.getProductPl(),
                request.getProductEn()
        );
    }

    @DeleteMapping("mappings/{projectId:.*}")
    public void delete(ProjectId projectId) {
        productsStorage.deleteMapping(projectId);
    }


    @Deprecated
    @SneakyThrows
    @RequestMapping(value = "/plugins", method = POST)
    public List<Plugin> list(@RequestBody StoreCriteria criteria, SecurityContextHolderAwareRequestWrapper request) {
        return new CapabilitiesResolver(getCapabilities(criteria.getClientCapabilities()))
                .query(channels(criteria, request));
    }

    @SneakyThrows
    @RequestMapping(value = "/plugins/{pluginId:.*}", method = POST)
    public ResponseEntity<Plugin> resolveSingle(
            @PathVariable String pluginId,
            @RequestBody ClientCapabilities capabilities,
            SecurityContextHolderAwareRequestWrapper request
    ) {
        log.info("Resolving plugin {} for {}", pluginId, capabilities);
        Project project = projectRepository.findOne(pluginId);
        if (project == null) {
            return notFound().build();
        }

        if (!hasAccessToProject(project, request)) {
            return status(FORBIDDEN).build();
        }

        UpdateChannel release = channels.get(ChannelId.of("release", pluginId));
        return ok(new CapabilitiesResolver(getCapabilities(capabilities)).query(release));
    }

    @GetMapping
    public List<com.suncode.autoupdate.server.client.api.Product> listProducts(Locale locale) {
        return store.list(locale);
    }

    @RequiredArgsConstructor
    private class CapabilitiesResolver {
        final Capabilities capabilities;


        Plugin query(UpdateChannel channel) {
            return channel.graph()
                    .availablePatches(ANY)
                    .getNewer()
                    .stream()
                    .reduce(new MatchingOrNewest(), MatchingOrNewest::considerNext, parallelNotSupported())
                    .getBest();
        }

        List<Plugin> query(Stream<UpdateChannel> channels) {
            return channels.map(channel -> channel.graph()
                    .availablePatches(ANY)
                    .getNewer()
                    .stream()
                    .reduce(new MatchingOrNewest(), MatchingOrNewest::considerNext, parallelNotSupported())
                    .get())
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .collect(toList());
        }

        @Getter
        private class MatchingOrNewest {
            Plugin best;

            Optional<Plugin> get() {
                return Optional.ofNullable(best);
            }

            MatchingOrNewest considerNext(Patch nextPatch) {
                if (best != null && best.isCompatible()) {
                    return this;
                }

                Plugin next = map(nextPatch);
                if (best == null) {
                    best = next;
                } else if (next.isCompatible()) {
                    best = next.withNewestVersion(best.getVersion());
                }
                return this;
            }
        }

        Plugin map(Patch patch) {
            List<Unresolved> unresolved = requirements(patch).stream()
                    .map(requirement -> {
                        Resolution resolution = capabilities.resolve(requirement);
                        Unresolved.Error error = resolution.accept(new Visitor<Unresolved.Error>() {

                            @Override
                            public Unresolved.Error missing() {
                                return MISSING;
                            }

                            @Override
                            public Unresolved.Error versionMismatch(Version version) {
                                return VERSION_MISMATCH;
                            }

                            @Override
                            public Unresolved.Error notActive() {
                                return null;
                            }

                            @Override
                            public Unresolved.Error running() {
                                return null;
                            }

                            @Override
                            public Unresolved.Error licenseUnsatisfied() {
                                return null;
                            }
                        });
                        return new Unresolved(
                                requirement.getId(),
                                requirement.getVersion().toString(),
                                requirement.isMandatory(),
                                error);
                    })
                    .filter(compose(Objects::nonNull, Unresolved::getError))
                    .collect(toList());

            return new Plugin(
                    patch.getChannel().getId().getProjectName(),
                    patch.getToVersion().toString(),
                    patch.getToVersion().toString(),
                    linkTo(PatchController.class)
                            .slash("download")
                            .slash(patch.getId())
                            .toUri(),
                    unresolved);
        }
    }

    private Stream<UpdateChannel> channels(StoreCriteria criteria, SecurityContextHolderAwareRequestWrapper request) {
        return projectRepository.findByPatchFormat(PLUGIN)
                .stream()
                .filter(withAccessPredicate(request))
                .flatMap(project -> project.getChannels()
                        .stream()
                        .filter(channel -> criteria.getChannels().matches(channel.getId().getName()))
                        .map(channel -> channels.get(channel.getId())));
    }

    @SneakyThrows
    private List<Requirement> requirements(Patch patch) {
        return Optional.ofNullable(patch.getProperty(REQUIREMENTS))
                .map(this::read)
                .map(pluginRequirements -> pluginRequirements.stream()
                        .map(pluginRequirement -> Requirement.builder()
                                .reference(Reference.of(pluginRequirement.getId(), parse(pluginRequirement.getVersion())))
                                .build())
                        .collect(toList()))
                .orElseGet(Collections::emptyList);
    }

    @SneakyThrows
    private Set<PluginRequirement> read(String prop) {
        return objectMapper.readValue(prop, new TypeReference<Set<PluginRequirement>>() {
        });
    }

    private static Capabilities getCapabilities(ClientCapabilities client) {
        return Capabilities.builder()
                .hostCapabilities(map(client.getHost(), hostCapability()))
                .pluginCapabilities(map(client.getPlugins(), pluginCapability()))
                .build();
    }

    private static Function<Component, PluginCapability> pluginCapability() {
        return component -> new PluginCapability(reference(component), ACTIVE);
    }

    private static Function<Component, HostCapability> hostCapability() {
        return component -> new HostCapability(reference(component));
    }

    private static Reference reference(Component component) {
        return Reference.of(component.getId(), parse(component.getVersion()));
    }

    @Value
    public static class AddRequest {
        ProjectId plugin;
        Product.Id productPl;
        Product.Id productEn;
    }

}
