package com.suncode.autoupdate.server.store;

import com.google.common.collect.ImmutableMap;
import com.suncode.autoupdate.server.client.api.Product;
import com.suncode.autoupdate.server.client.api.Product.Tag;
import lombok.RequiredArgsConstructor;
import lombok.experimental.FieldDefaults;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Locale;
import java.util.Locale.LanguageRange;
import java.util.Map;
import java.util.function.Function;

import static java.util.stream.Collectors.toList;
import static lombok.AccessLevel.PRIVATE;

@Component
@RequiredArgsConstructor
@FieldDefaults(level = PRIVATE, makeFinal = true)
public class Store {
    ProductsStorage productsStorage;

    private static final Locale PL = new Locale("pl");
    private static final Locale EN = new Locale("en");
    private final static Map<Locale, ProductExtractor> LOOKUP = ImmutableMap.of(
            PL, ProductExtractor.pl(),
            EN, ProductExtractor.en()
    );

    public List<Product> list(Locale locale) {
        Locale wantedLocale = Locale.lookup(
                LanguageRange.parse(locale.toLanguageTag()),
                LOOKUP.keySet()
        );

        ProductExtractor extractor = wantedLocale != null
                ? LOOKUP.get(wantedLocale)
                : ProductExtractor.en();

        return productsStorage.getMappedProducts().stream()
                .map(item -> {
                    com.suncode.autoupdate.server.store.wooapi.Product product = extractor.apply(item);
                    return new com.suncode.autoupdate.server.client.api.Product(
                            item.getProjectId(),
                            product.getSlug().asString(),
                            product.getName(),
                            product.getShortDescription(),
                            product.getPrice(),
                            product.getPriceHtml(),
                            product.getCategories().stream().map(tag -> new Tag(
                                    tag.getSlug(),
                                    tag.getName()
                            )).collect(toList()),
                            product.getTags().stream().map(tag -> new Tag(
                                    tag.getSlug(),
                                    tag.getName()
                            )).collect(toList()),
                            product.getPermalink(),
                            product.getImages().get(0).getSrc() // FIXME:
                    );
                })
                .collect(toList());
    }

    @FunctionalInterface
    private interface ProductExtractor extends Function<ProductsStorage.Mapped, com.suncode.autoupdate.server.store.wooapi.Product> {

        static ProductExtractor en() {
            return ProductsStorage.Mapped::getProductEn;
        }

        static ProductExtractor pl() {
            return ProductsStorage.Mapped::getProductPl;
        }
    }

}
