package com.suncode.autoupdate.server.patch;

import static com.google.common.collect.Lists.newArrayList;
import static org.springframework.http.MediaType.APPLICATION_OCTET_STREAM_VALUE;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPOutputStream;

import javax.servlet.http.HttpServletResponse;

import com.suncode.autoupdate.server.channel.Channel;
import com.suncode.autoupdate.server.channel.UpdateChannel;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.annotation.Secured;
import org.springframework.security.web.servletapi.SecurityContextHolderAwareRequestWrapper;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.autoupdate.server.patch.types.PlusWorkflowPatchFormat;
import com.suncode.autoupdate.patch.plusworkflow.archive.Index;
import com.suncode.autoupdate.server.channel.Channel.ChannelId;
import com.suncode.autoupdate.server.channel.Channels;
import com.suncode.autoupdate.server.client.Client;
import com.suncode.autoupdate.server.event.Events;
import com.suncode.autoupdate.server.patch.storage.PatchStorage;
import com.suncode.autoupdate.server.security.ClientAuth;
import com.suncode.autoupdate.server.security.HasAccess;

import lombok.Builder;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

/**
 * Patch REST API
 * 
 * @author Cezary Kozar 18 maj 2016
 */
@Slf4j
@RestController
@RequestMapping( "/patches" )
public class PatchController
{
    private final Events events;

    private final Channels channels;

    private final PatchStorage patchStorage;

    private final PatchRepository patchRepository;

    private ObjectMapper objectMapper = new ObjectMapper();

    public @Autowired PatchController( Events events, Channels channels, PatchStorage patchStorage,
                                       PatchRepository patchRepository )
    {
        this.events = events;
        this.channels = channels;
        this.patchStorage = patchStorage;
        this.patchRepository = patchRepository;
    }

    @RequestMapping( value = "download/{uuid}", method = RequestMethod.GET )
    public void download( @PathVariable String uuid, @RequestParam( required = false ) String clientEnv,
                          SecurityContextHolderAwareRequestWrapper request,
                          HttpServletResponse response )
        throws IOException
    {
        Patch patch = patchRepository.findOne( uuid );
        if ( patch == null )
        {
            throw new IllegalArgumentException( "No patch with given id" );
        }

        if ( patch.isArchived() )
        {
            throw new IllegalArgumentException( "Patch is archived" );
        }

        if ( !HasAccess.hasAccessToProject( patch.getChannel().getProject(), request ) )
        {
            throw new AccessDeniedException( "Client not permitted to download a patch" );
        }

        log.info( "Patch download requested: {}", patch.getId() );
        Resource patchResource = patchStorage.read( patch );
        events.patchDownload( patch, clientEnv );

        response.setHeader( "Content-Type", APPLICATION_OCTET_STREAM_VALUE );
        response.setHeader( "Content-Disposition", "attachment; filename=\"" + uuid + "\"" );

        try (InputStream in = patchResource.getInputStream();
                        OutputStream out = patchOutputStream( patch, response ))
        {
            IOUtils.copy( in, out );
        }
    }

    @SneakyThrows
    private OutputStream patchOutputStream( Patch patch, HttpServletResponse response )
    {
        OutputStream out = response.getOutputStream();
        Client client = ClientAuth.getAuthenticatedClient();
        boolean compress = client != null ? client.compressPatches() : false;
        return compress
                        ? new GZIPOutputStream( out )
                        : out;
    }

    @Secured("ROLE_ADMIN")
    @RequestMapping( value = "/{project:.+}/{channel}/upload", method = RequestMethod.POST )
    public void upload( @PathVariable String project,
                        @PathVariable String channel,
                        @RequestParam String fromVersion,
                        @RequestParam String toVersion,
                        @RequestParam( required = false ) String props,
                        @RequestParam MultipartFile patch )
        throws IOException
    {
        File patchFile = File.createTempFile( "patchdata-", ".upload" );
        try
        {
            Version from = new Version( fromVersion );
            Version to = new Version( toVersion );
            ChannelId channelId = ChannelId.of( channel, project );

            Map<String, String> properties = new HashMap<>();
            if ( props != null )
            {
                properties = objectMapper.readValue( props, new TypeReference<Map<String, String>>() {} );
            }

            patch.transferTo( patchFile );

            log.info( "Uploading patch to channel [{}] with properties [{}] ", channelId, properties );
            Patch uploaded = channels.get( channelId )
                .upload( from, to, patchFile, properties );

            events.patchUpload( uploaded );
        }
        finally
        {
            patchFile.delete();
        }
    }

    @RequestMapping( value = "{uuid}/diff", method = RequestMethod.GET )
    public Diff diff( @PathVariable String uuid, SecurityContextHolderAwareRequestWrapper request )
        throws IOException
    {
        Patch patch = patchRepository.findOne( uuid );
        if ( patch == null )
        {
            throw new IllegalArgumentException( "No patch with given id" );
        }

        if ( !HasAccess.hasAccessToProject( patch.getChannel().getProject(), request ) )
        {
            throw new AccessDeniedException( "Client not permitted to download a patch" );
        }

        PatchFormat patchFormat = patch.getChannel().getProject().getPatchFormat();
        if ( patchFormat == PatchFormat.PLUSWORKFLOW )
        {
            PlusWorkflowPatchFormat handler = (PlusWorkflowPatchFormat) patchFormat.handler();
            Index index = handler.readIndex( patchStorage.read( patch ).getFile() );
            return Diff.builder()
                .added( newArrayList( index.getAdded() ) )
                .updated( newArrayList( index.getUpdated() ) )
                .deleted( newArrayList( index.getDeleted() ) )
                .build()
                .sorted();
        }
        return null;
    }

    @Secured("ROLE_ADMIN")
    @RequestMapping( value = "/{uuid}", method = RequestMethod.DELETE )
    public void archive(@PathVariable String uuid) {
        Patch patch = patchRepository.findOne( uuid );
        if ( patch == null )
        {
            throw new IllegalArgumentException( "No patch with given id" );
        }

        patchRepository.save(patch.archive());
    }

    @Builder
    @Getter
    private static class Diff
    {
        List<String> added;
        List<String> updated;
        List<String> deleted;

        @JsonIgnore
        Diff sorted()
        {
            Collections.sort( added );
            Collections.sort( updated );
            Collections.sort( deleted );
            return this;
        }
    }
}
