package com.suncode.autoupdate.server.patch;

import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import javax.persistence.AttributeOverride;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Embedded;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;

import lombok.ToString;
import org.springframework.util.Assert;

import com.suncode.autoupdate.patch.PatchMeta;
import com.suncode.autoupdate.server.channel.Channel;

/**
 * Stored patch.
 * 
 * @author Cezary Kozar 18 maj 2016
 */
@Entity
@Table
@ToString(of = {"id", "fromVersion", "toVersion"})
public class Patch
{
    @Id
    private String id;

    @ManyToOne
    private Channel channel;

    @Embedded
    @AttributeOverride( name = "version", column = @Column( name = "fromVersion", nullable = false ) )
    private Version fromVersion;

    @Embedded
    @AttributeOverride( name = "version", column = @Column( name = "toVersion", nullable = false ) )
    private Version toVersion;

    @Temporal( TemporalType.TIMESTAMP )
    @Column( nullable = false )
    private Date uploaded;
    
    @Column
    private String checksum;

    @Column
    private boolean archived;

    @OneToMany( cascade = CascadeType.ALL, fetch = FetchType.LAZY, mappedBy = "patch" )
    private Set<PatchProperty> properties = new HashSet<>();

    Patch()
    {
    }

    public Patch( UUID id, Channel channel, Version fromVersion, Version toVersion )
    {
        Assert.notNull( id );
        Assert.notNull( channel );
        Assert.notNull( toVersion );

        this.id = id.toString();
        this.channel = channel;
        this.fromVersion = fromVersion;
        this.toVersion = toVersion;
        this.uploaded = new Date();
    }

    public String getId()
    {
        return id;
    }

    public Channel getChannel()
    {
        return channel;
    }

    public Version getFromVersion()
    {
        return fromVersion;
    }

    public Version getToVersion()
    {
        return toVersion;
    }

    public Date getUploaded()
    {
        return uploaded;
    }
    
    public String getChecksum()
    {
        return checksum;
    }

    public boolean isArchived() {
        return archived;
    }

    public void setChecksum( String checksum )
    {
        this.checksum = checksum;
    }

    public PatchMeta asPatchMeta()
    {
        return new PatchMeta( id, fromVersion.getVersion(), toVersion.getVersion() );
    }

    public Set<PatchProperty> getProperties()
    {
        return properties;
    }

    public String getProperty( String name )
    {
        for ( PatchProperty property : properties )
        {
            if ( property.getKey().equals( name ) )
            {
                return property.getValue();
            }
        }
        return null;
    }

    public void addProperty( String key, String value )
    {
        this.properties.add( new PatchProperty( key, value, this ) );
    }

    public Patch archive() {
        this.archived = true;
        return this;
    }
}
