package com.suncode.autoupdate.server.client;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.security.access.annotation.Secured;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.suncode.autoupdate.server.client.dto.ClientDto;

/**
 * Patch REST API
 * 
 * @author Cezary Kozar 18 maj 2016
 */
@RestController
@RequestMapping( "/clients" )
@Secured("ROLE_ADMIN")
public class ClientController
{
    private final ClientRepository clientRepository;

    public @Autowired ClientController( ClientRepository clientRepository )
    {
        this.clientRepository = clientRepository;
    }

    @RequestMapping( value = "", method = RequestMethod.POST )
    public ClientDto add( @RequestBody ClientDto newClient )
    {
        Client client = new Client( newClient.getId(), newClient.getName() );
        client.generateApiToken();
        clientRepository.save( client );

        return new ClientDto( client );
    }

    @RequestMapping( value = "", method = RequestMethod.GET )
    public List<ClientDto> list()
    {
        List<ClientDto> result = new ArrayList<>();
        for ( Client client : clientRepository.findAll() )
        {
            result.add( new ClientDto( client ) );
        }
        return result;
    }

    @RequestMapping( value = "/{id}", method = RequestMethod.GET )
    public ClientDto get( @PathVariable String id )
    {
        Client client = clientRepository.findOne( id );
        if ( client != null )
        {
            return new ClientDto( client );
        }
        throw new IllegalArgumentException( "No client with id [" + id + "]" );
    }

    @RequestMapping( value = "/{id}", method = RequestMethod.PUT )
    public ClientDto update( @PathVariable String id, @RequestBody ClientDto clientDto )
    {
        Client client = clientRepository.findOne( id );
        if ( client != null )
        {
            client.setName( clientDto.getName() );
            client.setCompressPatches( clientDto.isCompressPatches() );
            client.getEnvironments().clear();
            for ( String env : clientDto.getEnvironments() )
            {
                client.getEnvironments().add( new Environment( env ) );
            }
            clientRepository.save( client );
            return new ClientDto( client );
        }
        throw new IllegalArgumentException( "No client with id [" + id + "]" );
    }

    @RequestMapping( value = "/{id}", method = RequestMethod.DELETE)
    public void delete( @PathVariable String id )
    {
        clientRepository.delete( id );
    }
    
    @RequestMapping( value = "/{id}/token/refresh", method = RequestMethod.PUT )
    public String refreshToken( @PathVariable String id )
    {
        Client client = clientRepository.findOne( id );
        if ( client != null )
        {
            client.generateApiToken();
            clientRepository.save( client );
            return client.getApiToken();
        }
        throw new IllegalArgumentException( "No client with id [" + id + "]" );
    }

    @RequestMapping( value = "/{id}/config", method = RequestMethod.GET )
    public Resource generateConfig( @PathVariable String id, @RequestParam String project, @RequestParam String channel,
                                    @RequestParam String env, @RequestParam String path )
    {
        Client client = clientRepository.findOne( id );

        StringBuilder config = new StringBuilder();
        config.append( "# Project section" );
        config.append( "\nupdate.project=" + project );
        config.append( "\nupdate.channel=" + channel );
        config.append( "\nupdate.app.location=" + StringUtils.cleanPath( path ) );

        config.append( "\n\n# Client section" );
        config.append( "\nupdate.client.env=" + env );
        config.append( "\nupdate.client.apiToken=" + client.getApiToken() );

        byte[] data = config.toString().getBytes( StandardCharsets.UTF_8 );
        ByteArrayResource resource = new ByteArrayResource( data );
        return resource;
    }
}
