package com.suncode.autoupdate.server.channel.graph;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.suncode.autoupdate.server.channel.UpdateChannel;
import com.suncode.autoupdate.server.channel.graph.AvailablePatches.AvailablePatchesBuilder;
import com.suncode.autoupdate.server.patch.Patch;
import com.suncode.autoupdate.server.patch.PatchFormat;
import com.suncode.autoupdate.server.patch.PatchHandler;
import com.suncode.autoupdate.server.patch.Version;
import org.jgrapht.alg.CycleDetector;
import org.jgrapht.alg.DijkstraShortestPath;
import org.jgrapht.graph.DefaultDirectedGraph;
import org.jgrapht.traverse.DepthFirstIterator;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.suncode.autoupdate.server.channel.graph.Traversal.traverseEdges;
import static java.util.Collections.emptyList;
import static java.util.Comparator.comparing;

@SuppressWarnings("serial")
public class ChannelGraph
        extends DefaultDirectedGraph<Version, Patch> {
    private final UpdateChannel channel;

    public ChannelGraph(UpdateChannel channel) {
        super(Patch.class);

        Assert.notNull(channel);
        this.channel = channel;

        init();
    }

    private void init() {
        addVertex(Version.ANY);
        for (Patch patch : channel.getPatches()) {
            if(!patch.isArchived()) {
                Version from = patch.getFromVersion();
                Version to = patch.getToVersion();

                addVertex(from);
                addVertex(to);
                addEdge(from, to, patch);
            }
        }
    }

    public Patch findNewest() {
        return findNewest(null);
    }

    public Patch findNewest(Version source) {
        return doFindNewest(doFindNewest(null, source), Version.ANY);
    }

    private Patch doFindNewest(Patch newest, Version source) {
        if (source != null && !this.containsVertex(source)) {
            return null;
        }

        DepthFirstIterator<Version, Patch> iterator = new DepthFirstIterator<>(this, source);
        while (iterator.hasNext()) {
            Version v = iterator.next();
            if (outgoingEdgesOf(v).isEmpty()) {
                newest = newestPatch(newest, incomingEdgesOf(v));
            }
        }
        return newest;
    }

    public boolean willCreateCycle(Patch patch) {
        Version from = patch.getFromVersion();
        Version to = patch.getToVersion();
        try {
            addVertex(from);
            addVertex(to);
            addEdge(from, to, patch);

            CycleDetector<Version, Patch> cycleDetector = new CycleDetector<>(this);
            return cycleDetector.detectCycles();
        } finally {
            removeEdge(patch);
            if (edgesOf(from).isEmpty()) {
                removeVertex(from);
            }
            if (edgesOf(to).isEmpty()) {
                removeVertex(to);
            }
        }
    }

    public AvailablePatches availablePatches(Version version) {
        List<Patch> fromVersion = containsVertex(version) ? traverseEdges(this, version) : ImmutableList.of();
        List<Patch> fromAnyVersion = traverseEdges(this, Version.ANY);

        PatchHandler handler = channel.getChannel().getPatchFormat().handler();

        Comparator<Version> versionComparator = comparing(Version::getVersion, handler::compare);
        Comparator<Patch> patchComparator = comparing(Patch::getToVersion, versionComparator);

        List<Patch> patchesFromNewest = Stream.of(fromVersion, fromAnyVersion)
                .flatMap(Collection::stream)
                .distinct()
                .sorted(patchComparator.reversed())
                .collect(Collectors.toList());

        AvailablePatchesBuilder result = AvailablePatches.builder();
        patchesFromNewest.forEach(patch -> {
            if (version.equals(Version.ANY) || versionComparator.compare(patch.getToVersion(), version) > 0) {
                result.upgrade(patch);
            }
            else if(!patch.getToVersion().equals(version)){
                result.downgrade(patch);
            }
        });

        return result
                .newest(Iterables.getFirst(patchesFromNewest, null))
                .build();
    }

    private Patch newestPatch(Patch current, Set<Patch> edges) {
        Set<Patch> merged = new HashSet<>(edges);
        if (current != null) {
            merged.add(current);
        }
        return newestPatch(merged);
    }

    private Patch newestPatch(Set<Patch> edges) {
        Patch newest = null;
        for (Patch patch : edges) {
            newest = newest != null ? newestPatch(newest, patch) : patch;
        }
        return newest;
    }

    private Patch newestPatch(Patch patch1, Patch patch2) {
        PatchFormat patchFormat = channel.getChannel().getPatchFormat();
        int comparision = patchFormat.handler()
                .compare(patch1.getToVersion().getVersion(), patch2.getToVersion().getVersion());
        if (comparision > 0) {
            return patch1;
        }
        if (comparision < 0) {
            return patch2;
        }

        if (patch1.getUploaded().after(patch2.getUploaded())) {
            return patch1;
        }
        return patch2;
    }

    public List<Patch> shortestPath(Version from, Version to) {
        if (!containsVertex(from) || !containsVertex(to)) {
            return emptyList();
        }

        DijkstraShortestPath<Version, Patch> shortestPath = new DijkstraShortestPath<>(this, from, to);
        return Optional.ofNullable(shortestPath.getPathEdgeList())
                .orElse(emptyList());
    }
}