package com.suncode.autoupdate.server.channel;

import com.google.common.hash.Hashing;
import com.google.common.hash.HashingInputStream;
import com.suncode.autoupdate.server.patch.PatchHandler;
import com.suncode.autoupdate.server.channel.Channel.ChannelId;
import com.suncode.autoupdate.server.channel.graph.ChannelGraph;
import com.suncode.autoupdate.server.patch.Patch;
import com.suncode.autoupdate.server.patch.Version;
import com.suncode.autoupdate.server.patch.storage.PatchStorage;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.jgrapht.GraphPath;
import org.jgrapht.alg.DijkstraShortestPath;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.*;

@Slf4j
public class UpdateChannel
{
    @Getter
    private final Channel channel;

    private final ChannelRepository channelRepository;

    private final PatchStorage storage;

    private final ChannelGraph graph;

    public UpdateChannel( Channel channel, ChannelRepository channelRepository, PatchStorage storage )
    {
        Assert.notNull( channel );
        Assert.notNull( storage );
        Assert.notNull( channelRepository );

        this.channel = channel;
        this.storage = storage;
        this.channelRepository = channelRepository;

        this.graph = new ChannelGraph( this );
    }

    public ChannelId id()
    {
        return channel.getId();
    }

    public List<Patch> getPatches()
    {
        return Collections.unmodifiableList( channel.getPatches() );
    }

    public ChannelGraph graph()
    {
        return graph;
    }

    public Optional<Patch> newestPatch()
    {
        return Optional.ofNullable(graph.findNewest());
    }

    @SneakyThrows
    public UpdatePlan checkUpdates( Version version )
    {
        if ( !graph.containsVertex( version ) )
        {
            version = Version.ANY;
        }

        Patch newest = graph.findNewest( version );
        if ( newest == null )
        {
            log.info( "No newest version for {} in channel {}", version, channel );
            return UpdatePlan.EMPTY;
        }

        UpdatePlan plan = new UpdatePlan( newest.getToVersion() );
        if ( newest.getToVersion().equals( version ) )
        {
            log.info( "No updates found from {} to {} in channel {}", version, newest.getToVersion(), channel );
            return plan;
        }

        log.info( "Found newest version {} with ability to update from {} in channel {}", newest.getToVersion(),
                  version, channel );
        GraphPath<Version, Patch> updatePath = updatePath( version, newest.getToVersion() );
        for ( Patch patch : updatePath.getEdgeList() )
        {
            plan.addPatch( patch );

            if ( patch.getChecksum() == null )
            {
                Resource patchArchive = storage.read( patch );
                try (HashingInputStream in = new HashingInputStream( Hashing.md5(), patchArchive.getInputStream() ))
                {
                    byte[] buffer = new byte[1000000]; // 1MB buffer
                    while ( IOUtils.read( in, buffer ) == buffer.length )
                    {
                        // read just to calculate hash
                    }
                    patch.setChecksum( in.hash().toString() );
                }
            }
        }
        return plan;
    }

    private GraphPath<Version, Patch> updatePath( Version version, Version target )
    {
        // najkrótsza ścieżka
        DijkstraShortestPath<Version, Patch> shortestPath =
            new DijkstraShortestPath<Version, Patch>( graph, version, target );
        if ( shortestPath.getPath() == null )
        {
            shortestPath = new DijkstraShortestPath<Version, Patch>( graph, Version.ANY, target );
        }
        return shortestPath.getPath();
    }

    public synchronized Patch upload( Version fromVersion, Version toVersion, final File patchArchive,
                                      Map<String, String> properties )
        throws IOException
    {
        final Patch patch = getPatch( fromVersion, toVersion );
        for ( String key : properties.keySet() )
        {
            patch.addProperty( key, properties.get( key ) );
        }

        if ( graph.willCreateCycle( patch ) )
        {
            throw new IllegalStateException( "Cycle detected" );
        }

        final PatchHandler handler = channel.getPatchFormat().handler();
        if ( handler.transforms( patchArchive ) )
        {
            log.info( "Uploading patch {} to channel {} using {}", patch.getId(), channel, handler);
            
            storage.store( patch, new PatchStorage.StorageOutput() {
                @Override
                public void store( OutputStream out )
                    throws IOException
                {
                    handler.transform( patch.asPatchMeta(), patchArchive, out )
                            .forEach(patch::addProperty);
                }
            } );
            
            channel.getPatches().add( patch );
            channelRepository.save( channel );
            return patch;
        }
        throw new IllegalStateException( "Upload content not handled by " + handler );
    }

    private Patch getPatch( Version fromVersion, Version toVersion )
    {
        for ( Patch patch : channel.getPatches() )
        {
            if ( patch.getFromVersion().equals( fromVersion ) && patch.getToVersion().equals( toVersion ) )
            {
                return patch;
            }
        }
        return new Patch( UUID.randomUUID(), channel, fromVersion, toVersion );
    }
}
