package com.suncode.autoupdate.server.channel;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.suncode.autoupdate.server.channel.Channel.ChannelId;
import com.suncode.autoupdate.server.event.EventRepository;
import com.suncode.autoupdate.server.patch.Patch;
import com.suncode.autoupdate.server.patch.storage.PatchStorage;
import com.suncode.autoupdate.server.project.Project;
import com.suncode.autoupdate.server.project.ProjectRepository;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * Provides access to update channels.
 * 
 * @author Cezary Kozar 20 kwi 2016
 */
@Slf4j
@Service
@RequiredArgsConstructor( onConstructor = @__( @Autowired ) )
public class Channels
{
    private final ChannelRepository repository;
    private final ProjectRepository projectRepository;
    private final PatchStorage patchStorage;
    private final EventRepository eventRepository;

    @Transactional( readOnly = true )
    public UpdateChannel get( ChannelId id )
    {
        return channel( load( id ) );
    }

    @Transactional
    public UpdateChannel create( ChannelId id )
    {
        Project project = projectRepository.findOne( id.getProjectName() );
        if ( project == null )
        {
            throw new IllegalArgumentException( "Project [" + id.getProjectName() + "] not exists" );
        }

        Channel channel = new Channel( id.getName(), project );
        project.getChannels().add( channel );

        return channel( channel );
    }

    @Transactional
    public void remove( ChannelId id )
    {
        Channel channel = repository.findOne( id );
        for ( Patch patch : channel.getPatches() )
        {
            eventRepository.deleteByPatch( patch.getId() );
        }
        repository.delete( id );
        try
        {
            patchStorage.purge( id );
        }
        catch ( Exception e )
        {
            log.error( "Could not drop patches for channel {}", id, e );
        }
    }

    private UpdateChannel channel( Channel channel )
    {
        return new UpdateChannel( channel, repository, patchStorage );
    }

    private Channel load( ChannelId id )
    {
        Channel channel = repository.findOne( id );
        if ( channel != null )
        {
            return channel;
        }
        throw new IllegalStateException( "Channel [" + id + "] not found" );
    }
}
