import React from 'react';
import Table from './../../components/Table.jsx';
import Select2 from './../../components/Select2.jsx';
import DatePicker from './../../components/DatePicker.jsx';
import EventDetails from './EventDetails.jsx';
import moment from 'moment';

export default class Events extends React.Component {

    constructor(props) {
        super(props);
        this.state = {
            filter: {
                channel: "",
                project: "",
                client: "",
                patchType: [],
                dateFrom: "",
                dateTo: ""
            },
            selectedEvent: null
        };
        this.requestParams = this.requestParams.bind(this);
        this.responseData = this.responseData.bind(this);
        this.refreshGrid = debounce(this.refreshGrid.bind(this), 500);

        this.eventTypes = [
            'PATCH_DOWNLOAD',
            'PATCH_UPLOAD',
            'UPDATE_SUCCESS',
            'UPDATE_ERROR',
            'ROLLBACK_SUCCESS',
            'ROLLBACK_ERROR'
        ].map(item => {
            return {id: item, text: item}
        });
    }

    componentDidMount() {
        this.table = $("#events").DataTable({
            "paging": true,
            "info": false,
            "searching": false,
            "serverSide": true,
            "processing": true,
            "order": [
                [1, 'desc']
            ],
            "columns": [
                {
                    title: 'Zdarzenie',
                    data: 'type'
                }, {
                    title: 'Data',
                    data: 'date'
                }, {
                    title: 'Klient',
                    data: 'client'
                }, {
                    title: 'Środowisko',
                    data: 'clientEnv'
                }, {
                    title: 'Projekt',
                    data: 'project'
                }, {
                    title: 'Kanał',
                    data: 'channel'
                }, {
                    title: 'Wersja od',
                    data: 'fromVersion'
                }, {
                    title: 'Wersja do',
                    data: 'toVersion'
                }, {
                    title: 'PlusWorkflow',
                    data: 'plusworkflowVersion',
                    sortable: false
                }, {
                    sortable: false,
                    className: 'text-center',
                    width: 40,
                    render: function(data, type, full, meta) {
                        if (!full.patchId && !full.data) {
                            return '';
                        }
                        return "<a data-event-id=\"" + full.id + "\" class=\"event-details btn btn-default btn-xs txt-color-blue\">\
                        <i class=\"fa fa-search\"></i>\
                      </a>";
                    }
                }
            ],
            "ajax": {
                url: 'events',
                data: this.requestParams,
                dataSrc: this.responseData
            }
        });

        $('#events').on("click", '.event-details', this.eventDetailsClick.bind(this));
    }

    eventDetailsClick(e) {
        this.setState({
            selectedEvent: $(e.currentTarget).data('event-id')
        });
    }

    requestParams(data, settings) {
        var sort = data.order[0];
        var sortOverride = {
          "project": "patch.channel.project.name",
          "channel": "patch.channel.id.name",
          "fromVersion": "patch.fromVersion.version",
          "toVersion": "patch.toVersion.version"
        };
        var sortColumn = data.columns[sort.column].data;
        if(sortOverride.hasOwnProperty(sortColumn)){
          sortColumn = sortOverride[sortColumn];
        }
        return {
            filter: JSON.stringify(this.state.filter),
            size: data.length,
            page: data.start / data.length,
            sort: sortColumn + ',' + sort.dir
        };
    }

    responseData(data) {
        data.recordsTotal = data.totalElements;
        data.recordsFiltered = data.totalElements;
        return data.content.map(row => {
            return $.extend(row, {
                date: this.formatDate(new Date(row.date))
            });
        });
    }

    filterChange(prop, newValue) {
        var change = {};
        change[prop] = newValue;
        this.setState({filter: $.extend({}, this.state.filter, change)});
        this.refreshGrid();
    }

    refreshGrid() {
        this.table.ajax.reload();
    }

    render() {
        return (
            <div className="well well-light dataTables_wrapper">
                <div className="row">
                    <div className="col-md-12">
                        {this.renderSearchForm()}
                    </div>
                </div>

                <table id="events" className="table table-bordered table-hover"></table>
                <EventDetails event={this.state.selectedEvent} onClose={()=>{
                    this.setState({
                      selectedEvent: null
                    });
                  }}/>
            </div>
        )
    }

    renderSearchForm() {
        return (
            <form className="well form-horizontal">
                <legend>Filtrowanie</legend>
                <fieldset className="row">
                    <div className="col-md-6">
                        <div className="form-group">
                            <label className="col-md-3 control-label">Typ zdarzenia</label>
                            <div className="col-md-9">
                                <Select2 value={this.state.filter.patchType} values={this.eventTypes} onChange={val => {
                                    this.filterChange("patchType", val);
                                }}/>
                            </div>
                        </div>
                        <div className="form-group">
                            <label className="col-md-3 control-label">Data</label>
                            <div className="col-md-9">
                                <div className="row">
                                    <div className="col-md-6 ">
                                        <DatePicker value={this.state.filter.dateFrom} placeholder="Od" onChange={e => {
                                            this.filterChange("dateFrom", e.target.value);
                                        }}></DatePicker>
                                    </div>
                                    <div className="col-md-6 ">
                                        <DatePicker value={this.state.filter.dateTo} placeholder="Do" onChange={e => {
                                            this.filterChange("dateTo", e.target.value);
                                        }}></DatePicker>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div className="form-group">
                            <label className="col-md-3 control-label">Klient</label>
                            <div className="col-md-9">
                              <input type="text" className="form-control" value={this.state.filter.client} onChange={e => {
                                  this.filterChange("client", e.target.value)
                              }}/>
                            </div>
                        </div>
                        <div className="form-group">
                            <label className="col-md-3 control-label">Projekt</label>
                            <div className="col-md-9">
                                <input type="text" className="form-control" value={this.state.filter.project} onChange={e => {
                                    this.filterChange("project", e.target.value)
                                }}/>
                            </div>
                        </div>
                        <div className="form-group">
                            <label className="col-md-3 control-label">Kanał</label>
                            <div className="col-md-9">
                                <input type="text" className="form-control" value={this.state.filter.channel} onChange={e => {
                                    this.filterChange("channel", e.target.value)
                                }}/>
                            </div>
                        </div>
                    </div>
                </fieldset>
            </form>
        );
    }

    formatDate(d) {
      return moment(d).format('YYYY-MM-DD HH:mm:ss');
    }
}

function debounce(func, wait, immediate) {
    var timeout;

    return function() {
        var context = this,
            args = arguments;
        var callNow = immediate && !timeout;
        clearTimeout(timeout);

        timeout = setTimeout(function() {
            timeout = null;
            if (!immediate) {
                func.apply(context, args);
            }
        }, wait);
        if (callNow)
            func.apply(context, args);
        };
};
