package com.suncode.autoupdate.server.client.api;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.GZIPInputStream;

import org.apache.commons.io.IOUtils;

import com.google.common.hash.Hashing;
import com.google.common.hash.HashingInputStream;
import com.suncode.autoupdate.server.client.util.GZipUtil;

import okhttp3.ResponseBody;
import retrofit2.Response;
import retrofit2.Retrofit;

public class Patches
    extends Api<PatchesEndpoint>
{
    public Patches( Retrofit retrofit )
    {
        super( PatchesEndpoint.class, retrofit );
    }

    public void download( Patch patch, final OutputStream out )
        throws IOException
    {
        download( patch, new PatchStreamConsumer() {
            @Override
            public void accept( InputStream in )
                throws IOException
            {
                IOUtils.copy( in, out );
            }
        } );
    }

    public void download( Patch patch, PatchStreamConsumer out )
        throws IOException
    {
        Response<ResponseBody> archive = endpoint.download( patch.getId() ).execute();
        if ( archive.isSuccessful() )
        {
            try (HashingInputStream in = new HashingInputStream( Hashing.md5(), getUncompressed( archive ) ))
            {
                out.accept( in );
                String checksum = in.hash().toString();
                if ( !patch.getChecksum().equals( checksum ) )
                {
                    throw new IOException( String.format( "Patch [%s] checksum missmatch expected [%s] got [%s]", patch,
                                                          patch.getChecksum(), checksum ) );
                }
            }
            return;
        }
        throw new IOException( "Error downloading patch " + patch + ": " + archive.message() + " (" + archive.code()
            + ")" );
    }

    private InputStream getUncompressed( Response<ResponseBody> archive ) throws IOException
    {
        InputStream inputStream = new BufferedInputStream( archive.body().byteStream() );
        if ( GZipUtil.isGZipped( inputStream ) )
        {
            return new GZIPInputStream( inputStream );
        }
        return inputStream;
    }

    public interface PatchStreamConsumer
    {
        void accept( InputStream inputStream )
            throws IOException;
    }
}
