package com.suncode.autoupdate.server.client;

import com.suncode.autoupdate.server.client.api.*;
import lombok.AccessLevel;
import lombok.Builder;
import lombok.Getter;
import lombok.experimental.Accessors;
import lombok.experimental.FieldDefaults;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;
import retrofit2.Retrofit;

import java.io.IOException;
import java.net.URI;
import java.util.Optional;

@Getter
@Accessors(fluent = true)
@FieldDefaults( level = AccessLevel.PRIVATE )
public class UpdateServerClient
{
    final ApiToken apiToken;
    final Environment environment;

    Retrofit retrofit;
    Updates updates;
    Patches patches;
    Events events;
    Projects projects;
    TrialRegistration trialRegistration;
    Store store;

    private UpdateServerClient(URI updateServerAddress, ApiToken apiToken, Environment environment) {
        URI target = Optional.ofNullable(updateServerAddress)
                .orElseGet(() -> URI.create("https://www.aktualizacje.plusworkflow.pl"));

        this.apiToken = apiToken;
        this.environment = environment;
        this.retrofit = new RetrofitFactory().create(target, new AuthInterceptor() );
        this.updates = new Updates( retrofit );
        this.patches = new Patches( retrofit );
        this.events = new Events( retrofit );
        this.projects = new Projects( retrofit );
        this.trialRegistration = new TrialRegistration(retrofit);
        this.store = new Store(retrofit);
    }

    private class AuthInterceptor
        implements Interceptor
    {
        @Override
        public Response intercept( Chain chain )
            throws IOException
        {
            Request.Builder request = chain.request().newBuilder();
            HttpUrl.Builder url = chain.request().url().newBuilder();

            if ( apiToken != null )
            {
                request.addHeader( "X-Auth-Token", apiToken.getToken() );
            }
            if ( environment != null )
            {
                url.addQueryParameter( "clientEnv", environment.getEnv() );
            }
            return chain.proceed( request.url( url.build() ).build() );
        }
    }

    public static UpdateServerClient create()
    {
        return UpdateServerClient.builder().build();
    }

    public static UpdateServerClient create(Identity identity)
    {
        return UpdateServerClient.builder()
                .updateServerAddress(identity.getServiceUrl())
                .apiToken(identity.getApiToken())
                .environment(identity.getEnvironment())
                .build();
    }

    @Builder
    private static UpdateServerClient create( URI updateServerAddress, ApiToken apiToken, Environment environment )
    {
        return new UpdateServerClient( updateServerAddress, apiToken, environment );
    }
}
