package com.suncode.autoupdate.patcher.step;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;

import org.apache.commons.io.FilenameUtils;

import com.google.common.collect.Lists;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.suncode.autoupdate.patch.plusworkflow.ValidationResult;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patch.plusworkflow.archive.Checksum;
import com.suncode.autoupdate.patch.plusworkflow.archive.Index;
import com.suncode.autoupdate.patcher.Context;

import lombok.NonNull;
import lombok.SneakyThrows;
import lombok.Value;

@Value
public class Validator
{
    @NonNull
    Context context;
    
    @SneakyThrows
    public ValidationResult validate(Archive patch) {
        File project = context.getRoot();

        final ValidationResult result = new ValidationResult();

        final Checksum md5 = patch.getChecksum();
        final Index index = patch.getIndex();

        final Path projectRoot = project.toPath();
        final List<String> missing = Lists.newArrayList();
        for ( String path : md5.getPaths() )
        {
            if ( !isExcluded( path ) )
            {
                missing.add( path );
            }
        }

        Files.walkFileTree( projectRoot, new SimpleFileVisitor<Path>() {

            @Override
            public FileVisitResult visitFile( Path file, BasicFileAttributes attrs )
                throws IOException
            {
                Path relative = projectRoot.relativize( file );
                String relativePath = FilenameUtils.normalize( relative.toString(), true );

                // Usunięcie istniejących
                missing.remove( relativePath );

                if ( isExcluded( relativePath ) )
                {
                    return FileVisitResult.CONTINUE;
                }

                // 1. Plik który nie powinien istnieć
                if ( !md5.getPaths().contains( relativePath ) )
                {
                    result.getExtra().add( relativePath );
                    return FileVisitResult.CONTINUE;
                }

                // 2. Niepoprawny checksum
                HashCode expected = md5.get( relativePath );
                HashCode actual = com.google.common.io.Files.asByteSource( file.toFile() ).hash( Hashing.md5() );
                if ( !expected.equals( actual ) )
                {
                    // 2.1. Nowe
                    if ( index.isAdded( relativePath ) )
                    {
                        result.getChecksumAdded().add( relativePath );
                    }
                    // 2.2. Aktualizowane
                    else if ( index.isUpdated( relativePath ) )
                    {
                        result.getChecksumUpdated().add( relativePath );
                    }
                    // 2.3. Usuwane
                    else if ( index.isDeleted( relativePath ) )
                    {
                        result.getChecksumDeleted().add( relativePath );
                    }
                    // 2.4. Inne
                    else
                    {
                        result.getChecksum().add( relativePath );
                    }
                }
                return FileVisitResult.CONTINUE;
            };
        } );

        result.getMissing().addAll( missing );
        return result;
    }
    
    private static String[] EXCLUDES =
                    new String[] { ".autoupdate",
                                   ".patcher/",
                                   "META-INF/maven/",
                                   "META-INF/MANIFEST.MF" };
    
    private boolean isExcluded( String path )
    {
        for ( String exclude : EXCLUDES )
        {
            if ( path.startsWith( exclude ) )
            {
                return true;
            }
        }
        return false;
    }
}
