package com.suncode.autoupdate.patcher;

import com.google.common.base.Preconditions;
import com.suncode.autoupdate.patch.plusworkflow.ValidationResult;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patcher.cleanup.Cleaner;
import com.suncode.autoupdate.patcher.step.ApplyStep;
import com.suncode.autoupdate.patcher.step.Validator;
import com.suncode.autoupdate.server.client.UpdateServerClient;
import com.suncode.autoupdate.server.client.api.EventData;
import com.suncode.autoupdate.server.client.api.Events;
import lombok.NonNull;
import lombok.SneakyThrows;
import lombok.Value;

import java.io.File;
import java.util.Optional;
import java.util.UUID;

@Value
public class Rollbacks
{
    @NonNull
    Context context;
    @NonNull
    PatcherPlan plan;

    // TODO: refactor -- common part with Patcher
    @SneakyThrows
    public void applyRollback()
    {
        Validator validator = new Validator( context );
        ApplyStep applyStep = new ApplyStep( context );

        Events events = UpdateServerClient.builder()
            .apiToken( plan.getClient().getToken() )
            .environment( plan.getClient().getEnvironment() )
            .updateServerAddress( plan.getServerURI() )
            .build()
            .events();

        for ( String backupPath : plan.getRollbacks() )
        {
            doApplyBackup( backupPath, validator, applyStep, new SafeEventsSender(events) );
        }
        Cleaner cleaner = new Cleaner( context );
        cleaner.clean();
    }

    @SneakyThrows
    private void doApplyBackup( String backupPath, Validator validator, ApplyStep applyStep, SafeEventsSender events )
    {
        Optional<Archive> archive = readArchive( backupPath );
        if ( !archive.isPresent() )
        {
            throw new IllegalStateException( String
                .format( "Backup [%s] could not be found in backups dir. Patcher will exit.",
                         backupPath ) );
        }

        try (PatcherUnit unit = new PatcherUnit( archive.get() ))
        {
            unit.init();

            Archive patchArchive = unit.getArchive();
            UUID patchId = UUID.fromString( patchArchive.getMeta().getPatchId() );

            ValidationResult validation = null;
            try
            {
                validation = validator.validate( patchArchive );
                if ( !validation.valid() )
                {
                    throw new IllegalStateException( "Validation failed" );
                }

                Logger.info("Applying rollback [%s]", patchArchive.getMeta());
                applyStep.applyPatch( patchArchive );
                context.setCurrentVersion( patchArchive.getMeta().getToVersion() );
                Logger.info("Rollback applied [%s]", patchArchive.getMeta());

                events.rollbackSuccess( patchId, EventData.builder()
                    .success( true )
                    .validation( validation )
                    .build() );
            }
            catch ( Throwable t )
            {
                Logger.error("Rollback from file [%s] failed", backupPath);
                t.printStackTrace();
                events.rollbackError( patchId, EventData.builder()
                    .success( false )
                    .errorCause( t.getMessage() )
                    .validation( validation )
                    .build() );
            }
        }
    }

    private Optional<Archive> readArchive(String backup )
    {
        File archiveFile = new File( context.patcherDir(), "backups/" + backup );
        if ( archiveFile.exists() )
        {
            return Optional.of( new Archive( archiveFile ) );
        }
        return Optional.empty();
    }

    @Value
    private class PatcherUnit
        implements AutoCloseable
    {
        Archive archive;

        @SneakyThrows
        public void init()
        {
            archive.open();

            String startVersion = archive.getMeta().getFromVersion();
            Preconditions.checkState( context.getCurrentVersion().equals( startVersion ),
                                      String
                                          .format( "Patch [%s] updates system from version [%s], but current version is [%s]",
                                                   archive.getMeta().getPatchId(), startVersion,
                                                   context.getCurrentVersion() ) );
        }

        @Override
        public void close()
            throws Exception
        {
            archive.close();
        }
    }
}
