package com.suncode.autoupdate.patcher;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.suncode.autoupdate.server.client.ApiToken;
import com.suncode.autoupdate.server.client.Environment;
import lombok.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.InputStream;
import java.net.URI;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

@Data
@Builder
@AllArgsConstructor
@NoArgsConstructor
public class PatcherPlan {
    @NonNull
    URI serverURI;
    @NonNull
    String version;
    @NonNull
    List<UUID> patches;
    @NonNull
    List<String> rollbacks;
    @NonNull
    ClientContext client;

    Date applyAfter;

    public boolean shouldApply() {
        return applyAfter == null || applyAfter.before(new Date());
    }

    @Data
    @Builder
    @AllArgsConstructor
    @NoArgsConstructor
    public static class ClientContext {
        ApiToken token;
        Environment environment;
    }

    @SneakyThrows
    public void save(Context context) {
        File patcherPlanFile = patcherPlan(context);
        try (FileWriter writer = new FileWriter(patcherPlanFile)) {
            json().writeValue(writer, this);
        }
    }

    @SneakyThrows
    public static Optional<PatcherPlan> load(Context context) {
        File patcherPlanFile = patcherPlan(context);
        if (patcherPlanFile.exists()) {
            try (InputStream input = new FileInputStream(patcherPlanFile)) {
                return Optional.of(json().readValue(input, PatcherPlan.class));
            }
        }
        return Optional.empty();
    }

    private static File patcherPlan(Context context) {
        return new File(context.patcherDir(), "patcher.plan");
    }

    private static ObjectMapper json() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.enable(SerializationFeature.INDENT_OUTPUT);
        return mapper;
    }
}
