package com.suncode.autoupdate.patcher.step;

import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import com.suncode.autoupdate.patch.PatchMeta;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patch.plusworkflow.archive.Index;
import com.suncode.autoupdate.patch.plusworkflow.archive.Meta;
import com.suncode.autoupdate.patch.plusworkflow.archive.PatchAssembler;
import com.suncode.autoupdate.patcher.Context;
import com.suncode.autoupdate.patcher.Logger;
import lombok.NonNull;
import lombok.SneakyThrows;
import lombok.Value;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.function.Consumer;

import static java.nio.file.FileVisitResult.CONTINUE;
import static java.nio.file.FileVisitResult.SKIP_SUBTREE;
import static java.nio.file.Files.walkFileTree;

@Value
public class BackupStep {
    @NonNull
    Context context;

    public File createBackup(Archive archive)
            throws IOException {
        String prefix = uniqueName(archive);
        File backupDir = new File(context.patcherDir(), "backups");
        backupDir.mkdirs();
        File backup = new File(backupDir, prefix + ".backup");

        Logger.info("Creating backup of %s in %s", archive.getMeta(), backup.getAbsolutePath());
        generateBackup(context, backup, archive);
        return backup;
    }

    private void generateBackup(final Context context, File backup, final Archive archive)
            throws IOException {
        FileUtils.forceMkdir(backup.getParentFile());

        Meta meta = archive.getMeta();
        PatchMeta revered = new PatchMeta(meta.getPatchId(), meta.getToVersion(), meta.getFromVersion());

        Archive.assemble(revered, backup, patch -> doGenerate(context, archive, patch));
    }

    private void doGenerate(Context context, Archive archive, final PatchAssembler patch)
            throws IOException {
        final File root = context.getRoot();
        final Index index = archive.getIndex();

        // CHECKSUM
        walkFileTree(root.toPath(), new SimpleFileVisitor<Path>() {
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs)
                    throws IOException {
                if (file.endsWith(".autoupdate")) {
                    return CONTINUE;
                }

                String relative = FilenameUtils.normalize(root.toPath().relativize(file).toString(), true);
                if (!index.isAdded(relative) && !index.isUpdated(relative) && !index.isDeleted(relative)) {
                    try {
                        patch.checksum(relative, Files.asByteSource(file.toFile()).hash(Hashing.md5()));
                    } catch (Exception e) {
                        Logger.warn("Could not generate checksum for file %s", file.toAbsolutePath(), e);
                    }
                }
                return CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                return dir.toFile().equals(context.patcherDir())
                        ? SKIP_SUBTREE
                        : CONTINUE;
            }
        });

        for (String path : index.getChanged()) {

            File resolved = new File(root, path);
            index.changeFor(path).ifPresent(change -> {
                switch (change) {
                    case ADDED:
                        patch.delete(path);
                        patch.checksum(path, index.getChecksum(path));
                        break;
                    case UPDATED:
                        openOr(resolved,
                                stream -> {
                                    patch.update(path, stream);
                                    patch.checksum(path, index.getChecksum(path));
                                },
                                () -> {
                                    Logger.warn("Expected file %s to exists - will delete instead", path);
                                    patch.delete(path);
                                    patch.checksum(path, index.getChecksum(path));
                                }
                        );
                        break;
                    case DELETED:
                        openOr(resolved,
                                stream -> patch.add(path, stream),
                                () -> Logger.warn("Expected file %s to exists - ignoring.", path)
                        );
                        break;
                }
            });
        }
    }

    @SneakyThrows
    private void openOr(File file, Consumer<InputStream> consumer, Runnable fileNotFound) {
        try (InputStream in = new FileInputStream(file)) {
            consumer.accept(in);
        }
        catch (FileNotFoundException e) {
            fileNotFound.run();
        }
    }

    private String uniqueName(Archive archive) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd_HH_mm_ss");
        return archive.getMeta().getPatchId() + "_" + sdf.format(new Date());
    }
}
