package com.suncode.autoupdate.patcher;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.suncode.autoupdate.patch.plusworkflow.ValidationResult;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patcher.cleanup.Cleaner;
import com.suncode.autoupdate.patcher.step.ApplyStep;
import com.suncode.autoupdate.patcher.step.BackupStep;
import com.suncode.autoupdate.patcher.step.Validator;
import com.suncode.autoupdate.server.client.UpdateServerClient;
import com.suncode.autoupdate.server.client.api.EventData;
import com.suncode.autoupdate.server.client.api.Events;
import lombok.NonNull;
import lombok.SneakyThrows;
import lombok.Value;

import java.io.File;
import java.util.List;
import java.util.UUID;

@Value
public class Patcher
{
    @NonNull
    Context context;
    @NonNull
    PatcherPlan plan;

    @SneakyThrows
    public void applyPatch()
    {
        Validator validator = new Validator( context );
        BackupStep backup = new BackupStep( context );
        ApplyStep applyStep = new ApplyStep( context );

        SafeEventsSender events = new SafeEventsSender(UpdateServerClient.builder()
                        .apiToken(plan.getClient().getToken() )
                        .environment( plan.getClient().getEnvironment())
                        .updateServerAddress( plan.getServerURI() )
                        .build()
                        .events());

        for ( UUID patchId : plan.getPatches() )
        {
            List<ValidationResult> validationResultRef = Lists.newArrayList();
            try
            {
                doApplyPatch( patchId, validator, backup, applyStep, events, validationResultRef );
                events.updateSuccess( patchId, EventData.builder()
                                      .success( true )
                                      .validation( validationResultRef.get( 0 ) )
                                      .build() );
            }
            catch ( Throwable t )
            {
                Logger.error("Error applying patch %s", patchId, t);
                events.updateError( patchId, EventData.builder()
                                    .success( false )
                                    .errorCause( t.getMessage() )
                                    .build() );
            }
        }
        Cleaner cleaner = new Cleaner( context );
        cleaner.clean();
    }

    @SneakyThrows
    private void doApplyPatch( UUID patchId, Validator validator, BackupStep backup, ApplyStep applyStep, SafeEventsSender events, List<ValidationResult> validationResultRef )
    {
        Optional<Archive> archive = readArchive( patchId );
        if ( !archive.isPresent() )
        {
            throw new IllegalStateException(String.format( "Patch [%s] could not be found in pending dir. Patcher will exit.",
                                                           patchId ));
        }

        try (PatcherUnit unit = new PatcherUnit( archive.get() ))
        {
            unit.init();

            Archive patchArchive = unit.getArchive();
            ValidationResult validation = validator.validate( patchArchive );
            validationResultRef.add( validation );

            File backupFile = backup.createBackup( patchArchive );
            try
            {
                Logger.info("Applying patch [%s]", patchArchive.getMeta());
                applyStep.applyPatch( patchArchive );
                context.setCurrentVersion( patchArchive.getMeta().getToVersion() );
                Logger.info("Patch [%s] applied", patchArchive.getMeta());
            }
            catch ( Exception e )
            {
                Logger.error("Applying patch [%s] failed", patchId);

                Logger.info("Trying to rollback patch from backup [%s]", backupFile.getAbsolutePath());
                try (Archive backupArchive = new Archive( backupFile ))
                {
                    backupArchive.open();
                    applyStep.applyPatch( backupArchive );
                    context.setCurrentVersion( backupArchive.getMeta().getToVersion() );
                    events.rollbackSuccess( patchId, EventData.builder()
                                            .success( true )
                                            .build() );
                }
                catch ( Exception e1 )
                {
                    Logger.error("Rollback from file [%s] failed", backupFile.getAbsolutePath());
                    events.rollbackError( patchId, EventData.builder()
                                            .success( false )
                                            .errorCause( e1.getMessage() )
                                            .build() );
                }
                throw e;
            }
        }
    }

    private Optional<Archive> readArchive( UUID id )
    {
        File archiveFile = new File( context.patcherDir(), "pending/" + id.toString() );
        if ( archiveFile.exists() )
        {
            return Optional.of( new Archive( archiveFile ) );
        }
        return Optional.absent();
    }

    @Value
    private class PatcherUnit
        implements AutoCloseable
    {
        Archive archive;

        @SneakyThrows
        public void init()
        {
            archive.open();

            String startVersion = archive.getMeta().getFromVersion();
            Preconditions.checkState( context.getCurrentVersion().equals( startVersion ),
                                      String
                                          .format( "Patch [%s] updates system from version [%s], but current version is [%s]",
                                                   archive.getMeta().getPatchId(), startVersion,
                                                   context.getCurrentVersion() ) );
        }

        @Override
        public void close()
            throws Exception
        {
            archive.close();
        }
    }
}
