package com.suncode.autoupdate.plusworkflow.update.system;

import static java.util.Collections.emptyList;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.base.Preconditions;
import com.google.common.io.PatternFilenameFilter;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patch.plusworkflow.archive.Meta;
import com.suncode.autoupdate.patcher.Context;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
@RequiredArgsConstructor( onConstructor = @__( @Autowired ) )
public class Rollbacks
{
    private final Context context;

    public List<Rollback> resolve( Rollback rb )
    {
        List<Rollback> all = discover();
        List<Rollback> result = new ArrayList<>();
        boolean found = false;
        for ( Rollback rollback : all )
        {
            result.add( rollback );
            if ( rollback.equals( rb ) )
            {
                found = true;
                break;
            }
        }
        Preconditions.checkState( found );
        return result;
    }

    public List<Rollback> discover()
    {
        File backupDir = new File( context.getRoot(), ".patcher/backups" );
        if ( !backupDir.exists() )
        {
            return emptyList();
        }

        List<Rollback> rollbacks = new ArrayList<>();
        for ( File backupFile : backupDir.listFiles( BACKUP ) )
        {
            Rollback rollback = read( backupFile );
            if ( rollback != null )
            {
                rollbacks.add( rollback );
            }
        }

        Map<String, Rollback> filtered = new LinkedHashMap<>();
        String from = context.getCurrentVersion();
        while ( true )
        {
            Rollback found = newestRollback( from, rollbacks );
            if ( found == null || filtered.containsKey( found.getFileName() ) )
            {
                break;
            }
            filtered.put( found.getFileName(), found );
            from = found.getTo();
        }
        return new ArrayList<>( filtered.values() );
    }

    private Rollback newestRollback( String fromVersion, Collection<Rollback> rollbacks )
    {
        Rollback found = null;
        for ( Rollback rollback : rollbacks )
        {
            if ( rollback.getFrom().equals( fromVersion ) )
            {
                if ( found == null || rollback.getApplied().after( found.getApplied() ) )
                {
                    found = rollback;
                }
            }
        }
        return found;
    }

    private Rollback read( File file )
    {
        try (Archive archive = new Archive( file ))
        {
            archive.open();

            Meta meta = archive.getMeta();
            return Rollback.builder()
                .from( meta.getFromVersion() )
                .to( meta.getToVersion() )
                .fileName( file.getName() )
                .applied( new Date( file.lastModified() ) )
                .build();
        }
        catch ( Exception e )
        {
            log.error( "Failed to read rollbacks from {}", file, e );
            return null;
        }
    }

    private static final PatternFilenameFilter BACKUP = new PatternFilenameFilter( ".*\\.backup" );

}
