package com.suncode.autoupdate.plusworkflow.update.plugin;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.suncode.autoupdate.plusworkflow.update.Patches;
import com.suncode.autoupdate.server.client.api.Patch;
import com.suncode.pwfl.util.Streams;
import lombok.Getter;
import lombok.Value;
import lombok.experimental.Delegate;

import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Collections.emptyList;

@Value
public class PluginUpdates implements Patches {
    Patch newest;
    List<Patch> updates;
    List<Patch> downgrades;

    Function<Patch, Validated> validator;

    public List<Validated> getUpdates() {
        return updates.stream().map(validator).collect(Collectors.toList());
    }

    public List<Validated> getDowngrades() {
        return downgrades.stream().map(validator).collect(Collectors.toList());
    }

    @Override
    public Optional<Patch> get(UUID patchId) {
        return Stream.concat(updates.stream(), downgrades.stream())
                .filter(patch -> patch.getId().equals(patchId))
                .findFirst();
    }

    @Override
    public boolean hasAny() {
        return !updates.isEmpty();
    }

    @Override
    public Optional<String> getNewestVersion() {
        return  getUpdates().stream()
                .filter(Validated::isValid)
                .findFirst()
                .map(Validated::getToVersion);
    }

    @Override
    public List<Patch> neededFor(String version) {
        return Stream.of(updates, downgrades)
                .flatMap(List::stream)
                .filter(patch -> patch.getToVersion().equals(version))
                .limit(1)
                .reduce(Streams.noMoreThanOne())
                .map(Collections::singletonList)
                .orElse(emptyList());
    }

    @Value
    public static class Validated {
        @Delegate
        @Getter(onMethod = @__(@JsonIgnore))
        Patch patch;
        ValidationView validation;

        @JsonInclude
        public boolean isValid() {
            return validation.isValid();
        }
    }
}
