package com.suncode.autoupdate.plusworkflow.update.plugin;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.ImmutableMap;
import com.suncode.autoupdate.patch.plugin.PluginPatchProperties.PluginRequirement;
import com.suncode.autoupdate.plusworkflow.update.Changelog;
import com.suncode.autoupdate.plusworkflow.update.Patches;
import com.suncode.autoupdate.plusworkflow.update.PendingPatch;
import com.suncode.autoupdate.plusworkflow.update.download.DownloadQueue;
import com.suncode.autoupdate.plusworkflow.update.download.Downloads;
import com.suncode.autoupdate.plusworkflow.update.engine.UpdateEngine;
import com.suncode.autoupdate.plusworkflow.update.plugin.PluginUpdates.Validated;
import com.suncode.autoupdate.plusworkflow.update.support.AbstractComponentUpdate;
import com.suncode.autoupdate.plusworkflow.update.support.StateSummaryMapper;
import com.suncode.autoupdate.plusworkflow.update.system.Rollback;
import com.suncode.autoupdate.plusworkflow.util.Json;
import com.suncode.autoupdate.server.client.UpdateServerClient;
import com.suncode.autoupdate.server.client.api.AvailablePatches;
import com.suncode.autoupdate.server.client.api.Patch;
import com.suncode.plugin.framework.Plugin;
import com.suncode.plugin.framework.PluginControl;
import com.suncode.plugin.framework.PluginStore;
import com.suncode.plugin.framework.PluginStoreResource;
import com.suncode.plugin.framework.Version;
import com.suncode.plugin.framework.requirements.Requirement;
import com.suncode.plugin.framework.requirements.Requirements;
import com.suncode.pwfl.administration.user.UserContext;
import com.suncode.pwfl.audit.builder.ManualAuditBuilder;
import com.suncode.pwfl.audit.util.AuditTypes;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.experimental.FieldDefaults;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static com.suncode.autoupdate.patch.plugin.PluginPatchProperties.REQUIREMENTS;
import static com.suncode.autoupdate.plusworkflow.update.plugin.PluginsUpdater.RELEASE_CHANNEL;
import static com.suncode.autoupdate.plusworkflow.util.Exceptions.safe;
import static java.util.stream.Collectors.toSet;
import static lombok.AccessLevel.PRIVATE;

@FieldDefaults(level = PRIVATE, makeFinal = true)
public class PluginUpdate
        extends AbstractComponentUpdate {
    Plugin plugin;
    PluginControl control;
    String channel;

    public PluginUpdate(Plugin plugin, PluginStore store, String channel, UpdateEngine engine,
                        DownloadQueue downloadQueue,
                        StateSummaryMapper summaryMapper, PluginControl control) {
        super(store, engine, downloadQueue, summaryMapper);

        this.plugin = plugin;
        this.control = control;
        this.channel = channel;
    }

    @Override
    protected boolean autoConfirm() {
        return true;
    }

    @Override
    protected boolean needsValidation() {
        return false;
    }

    @Override
    protected boolean persistentState() {
        return false;
    }

    @Override
    protected boolean validate(List<PendingPatch> patches, Downloads download) {
        return true;
    }

    @Override
    public String key() {
        return plugin.getKey();
    }

    @Override
    protected String getProjectName() {
        return key();
    }

    @Override
    protected String getProjectDisplayName() {
        return plugin.getName();
    }

    @Override
    protected String getChannelName() {
        return channel;
    }

    @Override
    protected String getVersion() {
        return plugin.getVersion();
    }

    @Override
    public Changelog getPendingChangelog() {
        return Changelog.none();
    }

    @SneakyThrows
    private Requirements readRequirements(String raw) {
        if(raw == null) {
            return Requirements.none();
        }

        Set<PluginRequirement> requirements = Json.get().readValue(raw, new TypeReference<Set<PluginRequirement>>() {
        });

        return Requirements.of( requirements.stream()
                                    .map( req -> {
                                        boolean optional = !req.isMandatory();
                                        List<Version> versions = Arrays.asList( Version.parse( req.getVersion() ) );
                                        return new Requirement( req.getId(), versions, optional );
                                    } )
                                    .collect( toSet() ) );
    }

    @Override
    protected Patches getAvailablePatches(UpdateServerClient client) {
        AvailablePatches patches = client.updates().checkAll(
                getProjectName(),
                getChannelName(),
                getVersion());

        return new PluginUpdates(
                patches.getNewest(),
                patches.getNewer(),
                patches.getOlder(),
                this::validated);
    }

    private Validated validated(Patch patch) {
        Requirements requirements = readRequirements(patch.getProperties().get(REQUIREMENTS));
        PluginControl.Validation validation = control.validate(plugin,
                Version.parse(patch.getToVersion()), requirements);

        return new Validated(patch, new ValidationView(validation));
    }

    @SneakyThrows
    @Override
    protected boolean applyUpdates( List<PendingPatch> patches, Downloads download) {
        checkArgument(patches.size() == 1);
        PendingPatch patch = patches.get(0);
        PluginStoreResource resource = download.patchArchive(patch);

        Date started = new Date();
        Map<String, Object> params = ImmutableMap.of(
            "PLUGINID", plugin.getKey(),
            "NEW_VERSION", patch.getToVersion(),
            "OLD_VERSION", plugin.getVersion(),
            "UPDATE_TYPE", "automatic"
        );

        File temp = File.createTempFile(key(), ".update");
        try {
            try (InputStream inputStream = resource.getInputStream();
                 OutputStream outputStream = new FileOutputStream(temp)) {
                IOUtils.copy(inputStream, outputStream);
            }

            safe(
                () -> plugin.update( temp ),
                e -> plugin.getVersion().equals( patch.getToVersion() )
            );

            audit( true, params, started );
        } catch ( Exception e ) {
            audit( false, params, started );
        }
        finally {
            temp.delete();
        }
        return false;
    }

    private static void audit( boolean success, Map<String, Object> params, Date started )
    {
        ManualAuditBuilder.getInstance()
            .type( AuditTypes.AUDIT_UPDATE_PLUGIN )
            .success( success )
            .params( params )
            .username( UserContext.current().getUser().getUserName() )
            .started( started )
            .build()
            .log();
    }

    @Override
    protected void applyRollback(Rollback rollback) {
        throw new UnsupportedOperationException();
    }


    @Component
    @RequiredArgsConstructor(onConstructor = @__(@Autowired))
    @FieldDefaults(level = PRIVATE, makeFinal = true)
    public static class Factory {
        PluginControl control;
        UpdateEngine engine;
        PluginStore store;
        DownloadQueue downloadQueue;
        StateSummaryMapper summaryMapper;

        public PluginUpdate releaseChannel(Plugin plugin) {
            return new PluginUpdate(
                    plugin,
                    store,
                    RELEASE_CHANNEL,
                    engine,
                    downloadQueue,
                    summaryMapper,
                    control
            );
        }

    }

    @Override
    public ApplyResult apply(InputStream patch) {
        throw new UnsupportedOperationException();
    }
}
