package com.suncode.autoupdate.plusworkflow.update.engine;

import com.google.common.collect.Maps;
import com.suncode.autoupdate.plusworkflow.config.Config;
import com.suncode.autoupdate.plusworkflow.config.CorruptedConfigException;
import com.suncode.autoupdate.plusworkflow.update.State;
import com.suncode.autoupdate.plusworkflow.update.Updates;
import com.suncode.autoupdate.plusworkflow.util.Consumer;
import com.suncode.autoupdate.server.client.UpdateServerClient;
import com.suncode.plugin.framework.PluginStore;
import com.suncode.plugin.framework.PluginStoreResource;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

@Slf4j
@Component
public class UpdateEngine
{
    @Autowired
    private PluginStore store;

    @Autowired
    private List<ComponentUpdater> updaters;

    @Getter
    volatile Config config;
    @Getter
    volatile UpdateServerClient client;

    public boolean isConfigured()
    {
        return client != null;
    }

    @SneakyThrows
    @PostConstruct
    public void restore()
    {
        PluginStoreResource resource = store.read( "config.json" );
        if ( resource != null )
        {
            try (InputStream inputStream = resource.getInputStream())
            {
                try {
                    applyConfiguration( Config.read(inputStream) );
                }
                catch (CorruptedConfigException e) {
                    log.info("Cannot restore saved config file: config will be reset", e);
                }
            }
        }
    }

    public void applyConfiguration(Config config )
    {
        config.save( new Consumer<InputStream>() {
            @SneakyThrows
            public void accept( InputStream inputStream )
            {
                store.store( "config.json", inputStream, true );
            }
        } );

        this.config = config;
        this.client = UpdateServerClient.builder()
            .apiToken( config.getApiToken() )
            .environment( config.getEnvironment() )
            .updateServerAddress( config.getServerUrl() )
            .build();
    }

    public State getState()
    {
        return State.builder()
            .config( config )
            .configured( client != null )
            .updates( getUpdates() )
            .build();
    }

    public ComponentUpdate update( String project )
    {
        for ( ComponentUpdater updater : updaters )
        {
            for ( ComponentUpdate component : updater.getComponents() )
            {
                if ( component.key().equals( project ) )
                {
                    return component;
                }
            }
        }
        throw new IllegalArgumentException( "No update for poject " + project );
    }

    private Map<String, Updates> getUpdates()
    {
        Map<String, Updates> merged = Maps.newHashMap();
        for ( ComponentUpdater updater : updaters )
        {
            for ( ComponentUpdate component : updater.getComponents() )
            {
                Updates updates = component.getUpdates();
                if ( updates != null )
                {
                    merged.put( component.key(), updates );
                }
            }
        }
        return merged;
    }
}
