package com.suncode.autoupdate.plusworkflow.update.system;

import com.github.oxo42.stateless4j.StateMachineConfig;
import com.suncode.autoupdate.patch.PatchMeta;
import com.suncode.autoupdate.patch.plusworkflow.Props;
import com.suncode.autoupdate.patch.plusworkflow.ValidationResult;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patcher.Context;
import com.suncode.autoupdate.patcher.PatcherPlan;
import com.suncode.autoupdate.patcher.PatcherPlan.ClientContext;
import com.suncode.autoupdate.patcher.step.Validator;
import com.suncode.autoupdate.plusworkflow.audit.Audit;
import com.suncode.autoupdate.plusworkflow.update.Changelog;
import com.suncode.autoupdate.plusworkflow.update.Patches;
import com.suncode.autoupdate.plusworkflow.update.PendingPatch;
import com.suncode.autoupdate.plusworkflow.update.UpdateEvent;
import com.suncode.autoupdate.plusworkflow.update.UpdateState;
import com.suncode.autoupdate.plusworkflow.update.download.DownloadQueue;
import com.suncode.autoupdate.plusworkflow.update.download.Downloader;
import com.suncode.autoupdate.plusworkflow.update.download.Downloads;
import com.suncode.autoupdate.plusworkflow.update.engine.UpdateEngine;
import com.suncode.autoupdate.plusworkflow.update.support.AbstractComponentUpdate;
import com.suncode.autoupdate.plusworkflow.update.support.StateSummaryMapper;
import com.suncode.autoupdate.server.client.UpdateServerClient;
import com.suncode.autoupdate.server.client.api.AvailableUpdates;
import com.suncode.plugin.framework.Plugin;
import com.suncode.pwfl.SystemVersion;
import com.suncode.pwfl.administration.user.UserContext;
import com.suncode.pwfl.audit.builder.ManualAuditBuilder;
import com.suncode.pwfl.util.Maps;
import lombok.SneakyThrows;
import lombok.experimental.FieldDefaults;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import javax.annotation.Nullable;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.suncode.autoupdate.plusworkflow.audit.Audit.SYSTEM_UPDATE_CANCELED;
import static com.suncode.autoupdate.plusworkflow.audit.Audit.SYSTEM_UPDATE_SCHEDULED;
import static com.suncode.autoupdate.plusworkflow.update.UpdateEvent.UPLOADED;
import static com.suncode.autoupdate.plusworkflow.update.UpdateState.ERROR;
import static com.suncode.autoupdate.plusworkflow.update.UpdateState.INITIAL;
import static com.suncode.autoupdate.plusworkflow.update.UpdateState.NO_UPDATES;
import static com.suncode.autoupdate.plusworkflow.update.UpdateState.UPDATES_AVAILABLE;
import static com.suncode.autoupdate.plusworkflow.update.UpdateState.UP_TO_DATE;
import static com.suncode.autoupdate.plusworkflow.update.UpdateState.VALIDATING;
import static java.util.Arrays.asList;
import static lombok.AccessLevel.PRIVATE;

@Slf4j
@Component
@FieldDefaults(level = PRIVATE, makeFinal = true)
public class SystemUpdate
        extends AbstractComponentUpdate {
    Context context;
    Plugin plugin;
    Rollbacks rollbacks;
    Downloader downloader;

    public @Autowired
    SystemUpdate(Plugin plugin,
                 Context context,
                 UpdateEngine engine,
                 DownloadQueue downloadQueue,
                 StateSummaryMapper summaryMapper,
                 Rollbacks rollbacks,
                 Downloader downloader) {
        super(plugin.getPluginStore(), engine, downloadQueue, summaryMapper);
        this.context = context;
        this.plugin = plugin;
        this.rollbacks = rollbacks;
        this.downloader = downloader;

        getContext().getProperties().put("systemVersion", SystemVersion.getVersion());
        getContext().getProperties().put("rollbacks", rollbacks.discover());

        UpdateState state = getStateMachine().getState();
        if (state == UpdateState.POSTPONED_BACKUP || state == UpdateState.POSTPONED_UPDATE) {
            if (!PatcherPlan.load(context).isPresent()) {
                getStateMachine().fire(UpdateEvent.CANCEL);
            }
        }
    }

    @Override
    protected boolean autoConfirm() {
        return false;
    }

    @Override
    protected boolean needsValidation() {
        return true;
    }

    @Override
    protected boolean persistentState() {
        return true;
    }

    @Override
    public String key() {
        return "plusworkflow";
    }

    @Override
    protected String getProjectName() {
        if (getEngine().isConfigured()) {
            return getEngine().getConfig().getSystemProject();
        }
        return null;
    }

    @Override
    protected String getProjectDisplayName() {
        return "PlusWorkflow";
    }

    @Override
    protected String getChannelName() {
        if (getEngine().isConfigured()) {
            return getEngine().getConfig().getSystemChannel();
        }
        return null;
    }

    @Override
    protected String getVersion() {
        if (context.hasVersion()) {
            return context.getCurrentVersion();
        } else {
            error(new IllegalStateException("Cannot resolve system version"));
            return null;
        }
    }

    @Override
    public Changelog getPendingChangelog() {
        return getContext().getPatchesToApply().stream()
                .map(this::readIndex)
                .map(index -> Changelog.builder()
                        .added(index.getAdded())
                        .deleted(index.getDeleted())
                        .updated(index.getUpdated())
                        .build())
                .reduce(Changelog.none(), Changelog::merge);
    }

    @Override
    protected Patches getAvailablePatches(UpdateServerClient client) {
        AvailableUpdates updates = client.updates().checkNewest(
                getProjectName(),
                getChannelName(),
                getVersion());

        return new InOrderPatches(
                updates.getNewestVersion(),
                updates.getPatches());
    }

    @Override
    @SneakyThrows
    protected boolean applyUpdates( List<PendingPatch> patches, Downloads download) {
        Date started = new Date();
        File patcherHome = new File(context.getRoot(), ".patcher");
        patcherHome.mkdirs();

        File pendingDir = new File(patcherHome, "pending");
        pendingDir.mkdirs();

        for (PendingPatch patch : patches) {
            File target = new File(pendingDir, patch.getId().toString());
            try (InputStream in = download.patchArchive(patch).getInputStream()) {

                try (FileOutputStream outputStream = new FileOutputStream(target)) {
                    IOUtils.copy(in, outputStream);
                }
            }
        }

        List<UUID> patchesIds = new ArrayList<>();
        for (PendingPatch patch : patches) {
            patchesIds.add(patch.getId());
        }

        PatcherPlan plan = PatcherPlan.builder()
                .client(ClientContext.builder()
                        .environment(getEngine().getConfig().getEnvironment())
                        .token(getEngine().getConfig().getApiToken())
                        .build())
                .serverURI(getEngine().getConfig().getServerUrl())
                .version(context.getCurrentVersion())
                .patches(patchesIds)
                .rollbacks(Collections.<String>emptyList())
                .applyAfter(getContext().getApplyAfter())
                .build();
        plan.save(context);

        Resource resource = plugin.getResource("patcher.jar");
        File patcherBin = new File(context.getRoot(), ".patcher/patcher.jar");
        try (InputStream inputStream = resource.getInputStream()) {
            FileUtils.copyInputStreamToFile(inputStream, patcherBin);
        }

        getContext().getProperties().put("applyAfter", plan.getApplyAfter());

        audit( SYSTEM_UPDATE_SCHEDULED, started,
               param( "autoupdate.audit.version-to", systemVersionWithRevision( getContext().getTargetPatch() ) ),
               param( "autoupdate.audit.version-from", SystemVersion.getVersionIdentifier() ),
               param( "autoupdate.audit.schedule-date", nullableParam( getContext().getApplyAfter()) )
        );

        getStateMachine().fire(UpdateEvent.APPLIED_POSTPONE);
        return true;
    }

    @SafeVarargs
    private final void audit( Audit type, Date started, Map.Entry<String, Object>... params )
    {
        ManualAuditBuilder.getInstance()
            .type( type.key() )
            .success( true )
            .params( Arrays.stream( params ).collect( Maps.toNullableMap() ) )
            .username( UserContext.current().getUser().getUserName() )
            .started( started )
            .build()
            .log();
    }

    private static Object nullableParam( @Nullable Date value) {
        return value != null ? value : "";
    }

    @SneakyThrows
    protected boolean validate( List<PendingPatch> patches, Downloads download) {
        Validator validator = new Validator(context);

        File temp = File.createTempFile("temp", ".patch");
        try {
            // TODO: validate every patch and merge
            PendingPatch patch = patches.get(0);
            try (InputStream in = download.patchArchive(patch).getInputStream()) {
                IOUtils.copy(in, new FileOutputStream(temp));
            }

            try (Archive archive = new Archive(temp)) {
                archive.open();

                ValidationResult validationResult = validator.validate(archive);
                log.info(validationResult.toFormattedString());
                getContext().getProperties().put("validationResult", validationResult);

                return validationResult.valid();
            }
        } finally {
            temp.delete();
        }
    }

    @Override
    @SneakyThrows
    protected void cancelPostPoned() {
        File patcherHome = new File(context.getRoot(), ".patcher");
        patcherHome.mkdirs();

        File pendingDir = new File(patcherHome, "pending");
        FileUtils.cleanDirectory(pendingDir);
        FileUtils.deleteQuietly(new File(patcherHome, "patcher.plan"));

        audit( SYSTEM_UPDATE_CANCELED, new Date(),
               param( "autoupdate.audit.version-to", systemVersionWithRevision( getContext().getTargetPatch() ) ),
               param( "autoupdate.audit.version-from", SystemVersion.getVersionIdentifier() )
        );
    }

    @SneakyThrows
    @Override
    protected void applyRollback(Rollback rollback) {
        Date started = new Date();
        Validator validator = new Validator(context);

        List<Rollback> order = rollbacks.resolve(rollback);

        File temp = File.createTempFile("temp", ".patch");
        try {
            try (InputStream in = new FileInputStream(order.get(0).resolve(context))) {
                IOUtils.copy(in, new FileOutputStream(temp));
            }

            try (Archive archive = new Archive(temp)) {
                archive.open();

                ValidationResult validationResult = validator.validate(archive);
                log.info("Rollback {} validation result: {}", rollback, validationResult.toFormattedString());
                if (!validationResult.valid()) {
                    throw new IllegalStateException("Validation failed when rolling back updates");
                }
            }

            File patcherHome = new File(context.getRoot(), ".patcher");
            patcherHome.mkdirs();

            List<String> rollbcs = new ArrayList<>();
            for (Rollback rb : order) {
                rollbcs.add(rb.getFileName());
            }

            PatcherPlan plan = PatcherPlan.builder()
                    .client(ClientContext.builder()
                            .environment(getEngine().getConfig().getEnvironment())
                            .token(getEngine().getConfig().getApiToken())
                            .build())
                    .serverURI(getEngine().getConfig().getServerUrl())
                    .version(context.getCurrentVersion())
                    .patches(Collections.<UUID>emptyList())
                    .rollbacks(rollbcs)
                    .build();
            plan.save(context);

            Resource resource = plugin.getResource("patcher.jar");
            File patcherBin = new File(context.getRoot(), ".patcher/patcher.jar");
            try (InputStream inputStream = resource.getInputStream()) {
                FileUtils.copyInputStreamToFile(inputStream, patcherBin);
            }

            audit( Audit.SYSTEM_ROLLBACK_SCHEDULED, started,
                param("autoupdate.audit.version-to", rollback.getTo()),
                param("autoupdate.audit.version-from", rollback.getFrom()),
                param("autoupdate.audit.backup", rollback.getFileName())
            );
        } finally {
            temp.delete();
        }
    }

    @Override
    protected void configureStateMachine(StateMachineConfig<UpdateState, UpdateEvent> config) {
        EnumSet.of(ERROR, INITIAL, UP_TO_DATE, NO_UPDATES, UPDATES_AVAILABLE)
                .forEach(state -> config
                        .configure(state)
                        .permit(UPLOADED, VALIDATING)
                );
    }

    @Override
    @SneakyThrows
    public ApplyResult apply(InputStream stream) {
        if (getStateMachine().canFire(UPLOADED)) {
            byte[] bytes = IOUtils.toByteArray(stream);

            PatchMeta patch = Archive.readMeta(new ByteArrayInputStream(bytes));
            if (!patch.getFromVersion().equals(getVersion())) {
                log.info("Invalid version uploaded: {} != {}", patch.getFromVersion(), getVersion() );
                return ApplyResult.invalidVersion(getVersion(), patch.getFromVersion());
            }

            downloader.upload(patch, new ByteArrayInputStream(bytes));

            getContext().setPatchesToApply(asList(
                generatedPatch( patch )
            ));

            getStateMachine().fire(UPLOADED);
            return ApplyResult.ok();
        }
        throw new IllegalStateException("Could not apply now: current state" + getStateMachine().getState());
    }

    private static PendingPatch generatedPatch( PatchMeta patch )
    {
        return new PendingPatch(
            UUID.fromString( patch.getPatchId() ),
            patch.getFromVersion(),
            patch.getToVersion(),
            Collections.emptyMap()
        );
    }

    private static String systemVersionWithRevision( PendingPatch targetPatch )
    {
        String systemVersion = targetPatch.getProperties()
            .getOrDefault( Props.PLUSWORKFLOW_VERSION.property(), "" );
        return systemVersion
            + "(#" + targetPatch.getToVersion() + ")";
    }

    private Map.Entry<String, Object> param(String key, Object value) {
        return Maps.entry( key, value );
    }
}
