package com.suncode.autoupdate.plusworkflow.update.support;

import com.github.oxo42.stateless4j.StateMachine;
import com.github.oxo42.stateless4j.StateMachineConfig;
import com.github.oxo42.stateless4j.delegates.Action1;
import com.github.oxo42.stateless4j.triggers.TriggerWithParameters1;
import com.suncode.autoupdate.patch.plusworkflow.archive.ArchiveUtils;
import com.suncode.autoupdate.patch.plusworkflow.archive.Index;
import com.suncode.autoupdate.plusworkflow.update.Patches;
import com.suncode.autoupdate.plusworkflow.update.PendingPatch;
import com.suncode.autoupdate.plusworkflow.update.UpdateEvent;
import com.suncode.autoupdate.plusworkflow.update.UpdateState;
import com.suncode.autoupdate.plusworkflow.update.Updates;
import com.suncode.autoupdate.plusworkflow.update.download.Download;
import com.suncode.autoupdate.plusworkflow.update.download.DownloadQueue;
import com.suncode.autoupdate.plusworkflow.update.download.Downloads;
import com.suncode.autoupdate.plusworkflow.update.engine.ComponentUpdate;
import com.suncode.autoupdate.plusworkflow.update.engine.UpdateEngine;
import com.suncode.autoupdate.plusworkflow.update.system.Rollback;
import com.suncode.autoupdate.plusworkflow.util.Consumer;
import com.suncode.autoupdate.plusworkflow.util.Safe;
import com.suncode.autoupdate.server.client.UpdateServerClient;
import com.suncode.autoupdate.server.client.api.Patch;
import com.suncode.autoupdate.server.client.api.Project;
import com.suncode.plugin.framework.PluginStore;
import com.suncode.plugin.framework.PluginStoreResource;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.experimental.FieldDefaults;
import lombok.extern.slf4j.Slf4j;

import java.util.Date;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.suncode.autoupdate.plusworkflow.update.UpdateEvent.*;
import static com.suncode.autoupdate.plusworkflow.update.UpdateState.*;
import static lombok.AccessLevel.PRIVATE;

@Slf4j
@FieldDefaults( level = PRIVATE, makeFinal = true )
public abstract class AbstractComponentUpdate
    implements ComponentUpdate
{
    @Getter
    UpdateEngine engine;
    @Getter
    UpdateContext context;
    DownloadQueue downloadQueue;
    @Getter
    StateMachine<UpdateState, UpdateEvent> stateMachine;
    protected StateSummaryMapper summaryMapper;
    Downloads downloads;

    public AbstractComponentUpdate(PluginStore store, UpdateEngine engine, DownloadQueue downloadQueue,
                                   StateSummaryMapper summaryMapper )
    {
        this.engine = engine;
        this.downloadQueue = downloadQueue;
        this.summaryMapper = summaryMapper;
        this.downloads = new Downloads( store );

        if ( persistentState() )
        {
            PersistState state = new PersistState( store );
            this.context = state.getPersistedContext();
            this.stateMachine = setup( state );
        }
        else
        {
            this.context = new UpdateContext();
            this.stateMachine = setup( null );
        }
    }

    protected StateMachine<UpdateState, UpdateEvent> setup( PersistState persistState )
    {
        StateMachineConfig<UpdateState, UpdateEvent> config = new StateMachineConfig<>();
        config.configure( ERROR )
            .permit( CHECK, CHECKING );

        config.configure( INITIAL )
            .permit( CHECK, CHECKING )
            .onEntry(() -> {
                 if(engine.isConfigured()) {
                     stateMachine.fire(CHECK);
                 }
            });

        config.configure( CHECKING )
            .onEntry( new CheckAction() )
            .permit( CHECK_NO_UPDATES, UP_TO_DATE )
            .permit( CHECK_UPDATES_AVAILABLE, UPDATES_AVAILABLE )
            .permit( CHECK_NO_CHANNEL, NO_UPDATES );

        config.configure( UPDATES_AVAILABLE )
            .onEntryFrom(APPLIED_NOT_NEWEST, this::check)
            .permit( CHECK, CHECKING )
            .permit( DOWNLOAD_UPDATES, DOWNLOADING )
            .permit( RESTORE_BACKUP, APPLYING_BACKUP );

        config.configure( UP_TO_DATE )
            .onEntryFrom(APPLIED_NEWEST, this::check)
            .permit( CHECK, CHECKING )
            .permit( DOWNLOAD_UPDATES, DOWNLOADING )
            .permit( RESTORE_BACKUP, APPLYING_BACKUP );

        config.configure( NO_UPDATES )
            .permit( CHECK, CHECKING );

        config.configure( DOWNLOADING )
            .onEntryFrom( DOWNLOAD_TRIGGER, new DownloadAction(), String.class )
            .permit( DOWNLOAD_COMPLETED, DOWNLOADED );

        if ( needsValidation() )
        {
            config.configure( DOWNLOADED )
                .onEntry(() -> stateMachine.fire(APPLY))
                .permit( APPLY, VALIDATING );

            config.configure( VALIDATING )
                .onEntry( new ValidateAction() )
                .permit( VALIDATION_SUCCESS, READY )
                .permit( VALIDATION_ERROR, VALIDATED_ERROR );

            config.configure( VALIDATED_ERROR )
                .permit( CONFIRM, APPLYING )
                .permit( CANCEL, INITIAL );
        }
        else
        {
            config.configure( DOWNLOADED )
                .onEntry(() -> stateMachine.fire(APPLY))
                .permit( APPLY, READY );
        }

        config.configure( READY )
            .permit( CONFIRM, APPLYING )
            .permit( CANCEL, INITIAL );

        if ( autoConfirm() )
        {
            config.configure( READY )
                .onEntry( () -> stateMachine.fire( CONFIRM ) );
        }

        config.configure( APPLYING )
            .onEntry( new ApplyAction() )
            .permit( APPLIED_POSTPONE, POSTPONED_UPDATE )
            .permit( APPLIED_NEWEST, UP_TO_DATE )
            .permit( APPLIED_NOT_NEWEST, UPDATES_AVAILABLE );

        config.configure( POSTPONED_UPDATE )
            .onExit( new ExitPostPonedAction() )
            .permit( CANCEL, INITIAL );

        config.configure( APPLYING_BACKUP )
            .onEntry( new ApplyBackupAction() )
            .permit( RESTORE_BACKUP, POSTPONED_BACKUP );

        config.configure( POSTPONED_BACKUP )
            .onExit( new ExitPostPonedAction() )
            .permit( CANCEL, INITIAL );

        for ( UpdateState state : UpdateState.values() )
        {
            if ( state != ERROR )
            {
                config.configure( state )
                    .permit( ERROR_OCCURRED, ERROR );
            }
        }

        configureStateMachine( config );

        if ( persistState != null )
        {
            return new StateMachine<>(persistState.call(), persistState, persistState,
                    config);
        }
        return new StateMachine<>(INITIAL, config);
    }

    protected void configureStateMachine(StateMachineConfig<UpdateState, UpdateEvent> config) {

    }

    @Override
    public Updates getUpdates()
    {
        return Updates.builder()
                .name( getProjectName() )
                .displayName( getProjectDisplayName() )
                .channel( getChannelName() )
                .currentVersion( getVersion() )
                .newestVersion( context.getNewestVersion().orElse(null) )
                .stateSummary( summaryMapper.summaryOf( state(), context) )
                .state( state() )
                .lastCheck( context.getLastCheck() )
                .updates(context.getPatches())
                .pendingPatches(context.getPatchesToApply())
                .properties( context.getProperties() )
                .build();
    }

    @Override
    public final UpdateState state()
    {
        return stateMachine.getState();
    }

    protected void error( Throwable error )
    {
        if ( stateMachine.canFire( ERROR_OCCURRED ) )
        {
            stateMachine.fire( ERROR_TRIGGER, error );
        }
    }

    protected abstract boolean needsValidation();

    protected abstract boolean autoConfirm();

    protected abstract boolean persistentState();

    protected abstract String getProjectName();

    protected abstract String getProjectDisplayName();

    protected abstract String getChannelName();

    protected abstract String getVersion();

    protected abstract boolean validate(List<PendingPatch> patches, Downloads download );

    protected abstract boolean applyUpdates( List<PendingPatch> patches, Downloads download );

    protected abstract void applyRollback( Rollback rollback );

    protected abstract Patches getAvailablePatches(UpdateServerClient client);

    protected void cancelPostPoned()
    {
    }

    protected Index readIndex(PendingPatch patch) {
        PluginStoreResource resource = downloads.patchArchive(patch);
        checkNotNull(resource, "Patch %s was not yet downloaded", patch);

        return Safe.withInputStream(resource, ArchiveUtils::readIndex);
    }

    @Override
    public void check()
    {
        if ( stateMachine.canFire( CHECK ) )
        {
            stateMachine.fire( CHECK );
        }
    }

    @Override
    public void updateTo(String version)
    {
        if(stateMachine.canFire(DOWNLOAD_UPDATES)) {
            stateMachine.fire(DOWNLOAD_TRIGGER, version);
        }
    }

    @Override
    public void confirm( Date when )
    {
        if ( stateMachine.canFire( CONFIRM ) )
        {
            context.setApplyAfter( when );
            stateMachine.fire( CONFIRM );
        }
    }

    @Override
    public void cancel()
    {
        stateMachine.fire( CANCEL );
    }

    @Override
    public void rollback( Rollback rollback )
    {
        context.setRollback( rollback );
        stateMachine.fire( UpdateEvent.RESTORE_BACKUP );
    }

    private static final TriggerWithParameters1<Throwable, UpdateState, UpdateEvent> ERROR_TRIGGER =
        new TriggerWithParameters1<>( ERROR_OCCURRED, Throwable.class );
    private static final TriggerWithParameters1<String, UpdateState, UpdateEvent> DOWNLOAD_TRIGGER =
        new TriggerWithParameters1<>( DOWNLOAD_UPDATES, String.class );

    private class CheckAction
        extends Action
    {
        @SneakyThrows
        @Override
        protected void run( UpdateState from, UpdateState to, UpdateEvent trigger )
        {
            UpdateServerClient client = engine.getClient();
            context.setLastCheck( new Date() );

            log.info("Checking for updates {}@{}:{}", getProjectName(), getChannelName(), getVersion());

            Project project = findProject( client );
            if(project == null) {
                log.info("Project {} not found", getProjectName());
                stateMachine.fire( CHECK_NO_CHANNEL );
                return;
            }

            if ( !project.getChannels().contains( getChannelName() ) )
            {
                log.info("Channel {} not found in project {}", getChannelName(), getProjectName());
                stateMachine.fire( CHECK_NO_CHANNEL );
                return;
            }

            context.setPatches(getAvailablePatches(client));
            if ( context.getPatches().hasAny() )
            {
                stateMachine.fire( CHECK_UPDATES_AVAILABLE );
            }
            else
            {
                stateMachine.fire( CHECK_NO_UPDATES );
            }
        }

        private Project findProject( UpdateServerClient client )
        {
            for ( Project project : client.projects().getProjects() )
            {
                if ( project.getName().equals( getProjectName() ) )
                {
                    return project;
                }
            }
            return null;
        }
    }

    private class DownloadAction implements Action1<String> {

        @Override
        public void doIt( String version )
        {
            try
            {
                List<Patch> patches = context.getPatches().neededFor(version);
                if(patches.isEmpty()) {
                    throw new IllegalStateException("No patch to version [" + version + "] found");
                }

                context.setPatchesToApply( PendingPatch.map(patches) );

                Download download = downloadQueue.schedule( patches, engine.getClient() );
                download.await();
                stateMachine.fire( UpdateEvent.DOWNLOAD_COMPLETED );
            }
            catch ( Throwable t )
            {
                log.error( "Error when downloading updates", t );
                if ( stateMachine.canFire( ERROR_OCCURRED ) )
                {
                    stateMachine.fire( ERROR_TRIGGER, t );
                }
            }
        }
    }

    private class ValidateAction
        implements com.github.oxo42.stateless4j.delegates.Action
    {
        @Override
        public void doIt()
        {
            try
            {
                if ( validate( context.getPatchesToApply(), downloads ) )
                {
                    stateMachine.fire( VALIDATION_SUCCESS );
                }
                else
                {
                    stateMachine.fire( VALIDATION_ERROR );
                }
            }
            catch ( Throwable t )
            {
                log.error( "Error when validating update", t );
                if ( stateMachine.canFire( ERROR_OCCURRED ) )
                {
                    stateMachine.fire( ERROR_TRIGGER, t );
                }
            }
        }
    }

    private class ApplyAction
        extends Action
    {

        @Override
        protected void run( UpdateState from, UpdateState to, UpdateEvent trigger )
        {
            if ( !applyUpdates( context.getPatchesToApply(), downloads ) )
            {
                if ( getVersion().equals(context.getNewestVersion().orElse(null)) )
                {
                    stateMachine.fire( APPLIED_NEWEST );
                }
                else
                {
                    stateMachine.fire( APPLIED_NOT_NEWEST );
                }
            }
        }
    }

    private class ApplyBackupAction
        extends Action
    {
        @Override
        protected void run( UpdateState from, UpdateState to, UpdateEvent trigger )
        {
            applyRollback( context.getRollback() );
            stateMachine.fire( RESTORE_BACKUP );
        }
    }

    private class ExitPostPonedAction
        extends Action
    {
        @Override
        protected void run( UpdateState from, UpdateState to, UpdateEvent trigger )
        {
            cancelPostPoned();
        }
    }

    protected abstract class Action
        extends AbstractAction
    {
        public Action()
        {
            super( new Consumer<Throwable>() {

                @Override
                public void accept( Throwable error )
                {
                    if ( stateMachine.canFire( ERROR_OCCURRED ) )
                    {
                        stateMachine.fire( ERROR_TRIGGER, error );
                    }
                }
            } );
        }
    }
}
