package com.suncode.autoupdate.plusworkflow.update;

import com.google.common.collect.Sets;
import lombok.Builder;
import lombok.Singular;
import lombok.Value;

import java.util.Set;
import java.util.stream.Stream;

import static com.suncode.autoupdate.plusworkflow.util.Predicates.not;
import static java.util.stream.Collectors.toSet;

@Value
@Builder
public class Changelog {
    @Singular("added")
    Set<String> added;
    @Singular("updated")
    Set<String> updated;
    @Singular("deleted")
    Set<String> deleted;

    public static Changelog none() {
        return Changelog.builder().build();
    }

    public boolean isDeleted(String path) {
        return deleted.contains(path);
    }

    public boolean isAdded(String path) {
        return added.contains(path);
    }

    public Changelog merge(Changelog next) {
        Set<String> added = Stream.of(this.added, next.getAdded())
                .flatMap(Set::stream)
                .filter(not(next::isDeleted))
                .filter(not(this::isDeleted))
                .collect(toSet());

        Set<String> maybeUpdated = this.deleted
                .stream()
                .filter(next::isAdded)
                .collect(toSet());

        Set<String> updated = Stream.of(this.updated, next.getUpdated())
                .flatMap(Set::stream)
                .filter(not(this::isAdded))
                .filter(not(next::isDeleted))
                .collect(toSet());

        Set<String> deleted = Stream.of(this.deleted, next.getDeleted())
                .flatMap(Set::stream)
                .filter(not(next::isAdded))
                .filter(not(this::isAdded))
                .collect(toSet());

        return Changelog.builder()
                .added(added)
                .updated(Sets.union(updated, maybeUpdated))
                .deleted(deleted)
                .build();
    }
}
