package com.suncode.autoupdate.plusworkflow.config;

import com.suncode.autoupdate.patcher.Context;
import com.suncode.autoupdate.plusworkflow.update.engine.UpdateEngine;
import com.suncode.autoupdate.server.client.UpdateServerClient;
import com.suncode.autoupdate.server.client.api.Client;
import com.suncode.pwfl.SystemVersion;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Function;

import static com.suncode.autoupdate.plusworkflow.util.Safe.safe;
import static org.apache.commons.lang.StringUtils.isBlank;

@Slf4j
@Component
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class AutoConfiguration {
    private final Context context;
    private final UpdateEngine engine;
    private final PropertiesConfigLoader configLoader;

    @PostConstruct
    public void configure() {
        if (engine.isConfigured()) {
            return;
        }

        safe(
                this::configureTrial,
                e -> {
                    log.info("Skipping auto-configuration as it failed with an error: " + e.getMessage());
                    log.debug("Skipping auto-configuration as it failed with an error: " + e.getMessage(), e);
                });
    }

    private void configureTrial() {
        Optional<Config> config = autoUpdateConfig();
        if (!config.isPresent()) {
            log.info("Skipping auto-configuration since no config available");
            return;
        }

        if (!context.hasVersion()) {
            Optional<String> revision = clientRevision();
            if (revision.isPresent()) {
                context.setCurrentVersion(revision.get());
            } else {
                log.info("Skipping auto-configuration since no revision information provided");
                return;
            }
        }

        Config finalConfig = withTrial(config.get());
        engine.applyConfiguration(finalConfig);
        log.info("Configured updates with config {} and version {}", finalConfig, context.getCurrentVersion());
    }

    private Config withTrial(Config config) {
        UpdateServerClient client = UpdateServerClient.create();
        if (config.getApiToken() == null) {
            Client trial = client.trialRegistration().registerTrial();
            log.info("Registered new TRIAL auto-update client {}", trial);
            return config.withApiToken(trial.getApiToken());
        }
        return config;
    }

    private Optional<String> clientRevision() {
        return mapResource("client-version.properties", inputStream -> {
            Properties properties = new Properties();
            try {
                properties.load(inputStream);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            String revision = properties.getProperty("revision");
            return isBlank(revision) ? null : revision.trim();
        });
    }

    private Optional<Config> autoUpdateConfig() {
        return mapResource("autoupdate.properties", configLoader::load);
    }

    private <T> Optional<T> mapResource(String path, Function<InputStream, T> mapper) {
        ClassPathResource resource = new ClassPathResource(path, SystemVersion.class.getClassLoader());
        if (resource.exists()) {
            try {
                try (InputStream in = resource.getInputStream()) {
                    return Optional.ofNullable(mapper.apply(in));
                }
            } catch (Exception e) {
                log.error("Could not read system resource {}", resource, e);
            }
        }
        return Optional.empty();
    }
}
