package com.suncode.autoupdate.patch.plusworkflow.pluspatch;

import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.Map.Entry;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import com.google.common.collect.MapDifference;
import com.google.common.collect.MapDifference.ValueDifference;
import com.google.common.collect.Maps;
import com.google.common.hash.HashCode;
import com.suncode.autoupdate.patch.PatchMeta;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patch.plusworkflow.archive.PatchAssembler;
import com.suncode.autoupdate.patch.plusworkflow.archive.PatchAssembler.Assemble;

/**
 * Transforms PlusPatch3 compatible archives to {@linkplain Archive}.
 * 
 * @author Cezary Kozar 26 kwi 2016
 */
public class PlusPatch3Transformer
{
    private final PatchMeta patch;

    private final File archive;

    public PlusPatch3Transformer( PatchMeta patch, File patchArchive )
    {
        this.patch = patch;
        this.archive = patchArchive;
    }

    public static boolean isPlusPatch3( File archive )
    {
        return PlusPatch3Spec.matches( archive );
    }

    public Archive transform()
        throws IOException
    {
        try (ZipFile zip = new ZipFile( archive ))
        {
            final Map<String, HashCode> checksumOld = PlusPatch3Spec.oldMd5( zip );
            final Map<String, HashCode> checksumNew = PlusPatch3Spec.newMd5( zip );
            final MapDifference<String, HashCode> diff = Maps.difference( checksumNew, checksumOld );

            return Archive.assemble( patch, new Assemble() {

                @Override
                public void assemble( PatchAssembler patch )
                    throws IOException
                {

                    // 1. Sumy kontrolne wszystkich plików
                    for ( Entry<String, HashCode> entry : checksumOld.entrySet() )
                    {
                        patch.checksum( entry.getKey(), entry.getValue() );
                    }

                    // 2. Nowe
                    for ( Entry<String, HashCode> entry : diff.entriesOnlyOnLeft().entrySet() )
                    {
                        String path = entry.getKey();
                        ZipEntry zipEntry = zip.getEntry( path );
                        patch.add( path, zip.getInputStream( zipEntry ), entry.getValue() );
                    }

                    // 3. Zaktualizowane
                    for ( Entry<String, ValueDifference<HashCode>> entry : diff.entriesDiffering().entrySet() )
                    {
                        String path = entry.getKey();
                        ZipEntry zipEntry = zip.getEntry( path );
                        patch.update( path, zip.getInputStream( zipEntry ), entry.getValue().leftValue() );
                    }

                    // 4. Usunięte
                    for ( Entry<String, HashCode> entry : diff.entriesOnlyOnRight().entrySet() )
                    {
                        patch.delete( entry.getKey() );
                    }
                }
            } );
        }
    }
}
