package com.suncode.autoupdate.patch.plusworkflow.archive.merge;

import com.google.common.hash.HashCode;
import com.suncode.autoupdate.patch.PatchMeta;
import com.suncode.autoupdate.patch.plusworkflow.archive.Archive;
import com.suncode.autoupdate.patch.plusworkflow.archive.Checksum;
import com.suncode.autoupdate.patch.plusworkflow.archive.Index;
import com.suncode.autoupdate.patch.plusworkflow.archive.PatchAssembler;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.Value;
import lombok.experimental.FieldDefaults;

import java.io.File;
import java.io.InputStream;
import java.util.*;
import java.util.function.BinaryOperator;
import java.util.function.Consumer;
import java.util.function.Function;

import static lombok.AccessLevel.PRIVATE;

@RequiredArgsConstructor
@FieldDefaults(level = PRIVATE, makeFinal = true)
public class ArchiveMerger {
    @NonNull
    Collection<Archive> archives;

    public Archive merge(File output) {
        try {
            Merger merged = archives.stream()
                    .peek(archive -> open(archive))
                    .sorted(Comparator.comparingInt(archive -> Integer.parseInt(archive.getMeta().getFromVersion())))
                    .reduce(
                            new Merger(), Merger::apply, noParallel()
                    );
            return merged.assemble(output);
        } finally {
            archives.forEach(archive -> close(archive));
        }
    }


    private class Merger {
        Checksum stateZeroChecksum;
        String from;
        String to;
        Map<String, Change> changes = new HashMap<>();

        @SneakyThrows
        public Merger apply(Archive archive) {
            if (stateZeroChecksum == null) {
                stateZeroChecksum = archive.getChecksum();
                from = archive.getMeta().getFromVersion();
            } else {
                if (!archive.getMeta().getFromVersion().equals(to)) {
                    throw new IllegalArgumentException(
                            String.format("Not connected patch %s -> %s -\\> %s -> %s",
                                    from, to, archive.getMeta().getFromVersion(), archive.getMeta().getToVersion()));
                }
            }
            to = archive.getMeta().getToVersion();

            Index index = archive.getIndex();
            Checksum checksum = archive.getChecksum();

            index.getAdded().forEach(addedPath -> {
                changes.compute(addedPath,
                        (path, change) -> {
                            if (change == null) {
                                if(stateZeroChecksum.get(path) != null) {
                                    throw new IllegalArgumentException("There is already file in stateZeroChecksum:" + path);
                                }
                                return new Add(path, index.getAddedChecksum(path), archive);
                            } else {
                                return change.merge(
                                        add -> notAllowed(),
                                        update -> notAllowed(),
                                        delete -> {
                                            HashCode stateZero = stateZeroChecksum.get(path);
                                            HashCode now = index.getAddedChecksum(path);
                                            if(now.equals(stateZero)) {
                                                return null;
                                            }
                                            else {
                                                return new Update(path, now, archive);
                                            }
                                        }
                                );
                            }
                        });
            });

            index.getUpdated().forEach(updatedPath -> {
                changes.compute(updatedPath,
                        (path, change) -> {
                            if (change == null) {
                                if(stateZeroChecksum.get(path) == null) {
                                    throw new IllegalArgumentException("There no file in stateZeroChecksum:" + path);
                                }
                                return new Update(path, index.getUpdatedChecksum(path), archive);
                            } else {
                                return change.merge(
                                        add -> {
                                            if (!add.getHashCode().equals(checksum.get(path))) {
                                                throw new IllegalStateException("Add confict");
                                            }
                                            return new Add(path, index.getUpdatedChecksum(path), archive);
                                        },
                                        update -> {
                                            if (!update.getHashCode().equals(checksum.get(path))) {
                                                throw new IllegalStateException("Add confict");
                                            }

                                            HashCode stateZero = stateZeroChecksum.get(path);
                                            HashCode now = index.getUpdatedChecksum(path);
                                            if(now.equals(stateZero)) {
                                                return null;
                                            }
                                            else {
                                                return new Update(path, now, archive);
                                            }
                                        },
                                        delete -> notAllowed()
                                );
                            }
                        });
            });

            index.getDeleted().forEach(deletedPaths -> {
                changes.compute(deletedPaths,
                        (path, change) -> {
                            if (change == null) {
                                if(stateZeroChecksum.get(path) == null) {
                                    throw new IllegalArgumentException("There is no file in stateZeroChecksum:" + path);
                                }
                                return new Delete(path);
                            } else {
                                return change.merge(
                                        add -> {
                                            return null;
                                        },
                                        update -> {
                                            if (!update.getHashCode().equals(checksum.get(path))) {
                                                throw new IllegalArgumentException("Add confict");
                                            }
                                            return new Delete(path);
                                        },
                                        delete -> new Delete(path)
                                );
                            }
                        });
            });
            return this;
        }

        private <T> T notAllowed() {
            throw new IllegalArgumentException("Add confict");
        }

        @SneakyThrows
        Archive assemble(File output) {
            // now use assembler to assemly archive
            try (PatchAssembler assembler = new PatchAssembler(
                    new PatchMeta(
                            UUID.randomUUID().toString(),
                            from,
                            to
                    ),
                    output
            )) {

                stateZeroChecksum.getPaths().forEach(
                        path -> assembler.checksum(path, stateZeroChecksum.get(path))
                );

                changes.forEach((path, change) -> change.visit(
                        add -> add.apply(assembler),
                        update -> update.apply(assembler),
                        delete -> delete.apply(assembler)
                ));
            }
            return new Archive(output);
        }
    }

    private interface Change {
        void apply(PatchAssembler assembler);

        <T> T merge(
                Function<Add, T> add,
                Function<Update, T> update,
                Function<Delete, T> delete
        );

        default void visit(
                Consumer<Add> add,
                Consumer<Update> update,
                Consumer<Delete> delete
        ) {
            merge(asFn(add), asFn(update), asFn(delete));
        }

        static <R, T> Function<R, T> asFn(Consumer<R> consumer) {
            return r -> {
                consumer.accept(r);
                return null;
            };
        }
    }

    @Value
    private class Add implements Change {
        String path;
        HashCode hashCode;
        Archive patch;

        @Override
        @SneakyThrows
        public void apply(PatchAssembler assembler) {
            try (InputStream in = patch.get(path)) {
                assembler.add(path, in);
            }
        }

        @Override
        public <T> T merge(
                Function<Add, T> add,
                Function<Update, T> update,
                Function<Delete, T> delete
        ) {
            return add.apply(this);
        }
    }

    @Value
    private class Update implements Change {
        String path;
        HashCode hashCode;
        Archive patch;

        @Override
        @SneakyThrows
        public void apply(PatchAssembler assembler) {
            try (InputStream in = patch.get(path)) {
                assembler.update(path, in);
            }
        }

        @Override
        public <T> T merge(
                Function<Add, T> add,
                Function<Update, T> update,
                Function<Delete, T> delete
        ) {
            return update.apply(this);
        }
    }

    @Value
    private class Delete implements Change {
        String path;

        @Override
        @SneakyThrows
        public void apply(PatchAssembler assembler) {
            assembler.delete(path);
        }

        @Override
        public <T> T merge(
                Function<Add, T> add,
                Function<Update, T> update,
                Function<Delete, T> delete
        ) {
            return delete.apply(this);
        }
    }

    @SneakyThrows
    private void open(Archive archive) {
        archive.open();
    }

    @SneakyThrows
    private void close(Archive archive) {
        archive.close();
    }

    private BinaryOperator<Merger> noParallel() {
        return (merger, merger2) -> {
            throw new UnsupportedOperationException();
        };
    }
}
