package com.suncode.autoupdate.patch.plusworkflow.archive;

import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.google.common.hash.HashingInputStream;
import com.suncode.autoupdate.patch.PatchMeta;
import lombok.SneakyThrows;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;

import java.io.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import static com.google.common.hash.Hashing.md5;

/**
 * Assembler used to create binary patch.
 *
 * @author Cezary Kozar 17 maj 2016
 */
public class PatchAssembler
    implements Closeable
{
    public interface Assemble
    {
        void assemble( PatchAssembler patch )
            throws IOException;
    }

    private final Meta meta;

    private final Index index = new Index();

    private final Checksum checksum = new Checksum();

    private final ZipOutputStream zip;

    public PatchAssembler( PatchMeta patch, File storage )
        throws FileNotFoundException
    {
        this.zip = new ZipOutputStream( new FileOutputStream( storage ) );
        this.meta = new Meta( patch );
    }

    public PatchAssembler checksum( String path, HashCode md5 )
    {
        checksum.add( normalize( path ), md5 );
        return this;
    }

    @SneakyThrows
    public PatchAssembler add( String path, InputStream input )
    {
        HashingInputStream hashing = new HashingInputStream(md5(), input);
        write( "data/" + path, hashing);
        index.added( normalize( path ), hashing.hash() );
        return this;
    }

    @Deprecated
    public PatchAssembler add( String path, InputStream input, HashCode md5 )
        throws IOException
    {
        write( "data/" + path, input );
        index.added( normalize( path ), md5 );
        return this;
    }

    @SneakyThrows
    public PatchAssembler update( String path, InputStream input )
    {
        HashingInputStream hashing = new HashingInputStream(md5(), input);
        write( "data/" + path, hashing);
        index.updated( normalize( path ), hashing.hash() );
        return this;
    }

    @Deprecated
    public PatchAssembler update( String path, InputStream input, HashCode md5 )
        throws IOException
    {
        write( "data/" + path, input );
        index.updated( normalize( path ), md5 );
        return this;
    }

    public PatchAssembler delete( String path )
    {
        index.deleted( normalize( path ) );
        return this;
    }

    private void write( String entry, InputStream input )
        throws IOException
    {
        zip.putNextEntry( new ZipEntry( entry ) );
        IOUtils.copy( input, zip );
    }

    private void write( ArchivePart part )
        throws IOException
    {
        zip.putNextEntry( new ZipEntry( part.location() ) );
        part.writeTo( zip );
    }

    private void write()
        throws IOException
    {
        write( meta );
        write( index );
        write( checksum );
    }

    @Override
    public void close()
        throws IOException
    {
        write();
        zip.close();
    }

    private String normalize(String path) {
        return FilenameUtils.normalize( path, true );
    }
}