package com.suncode.autoupdate.patch.plusworkflow.archive;

import com.google.common.collect.ImmutableSet;
import com.google.common.hash.HashCode;

import java.util.*;

public class Index
        extends ArchivePart {
    private Map<String, HashCode> added = new LinkedHashMap<>();

    private Map<String, HashCode> updated = new LinkedHashMap<>();

    private Set<String> deleted = new LinkedHashSet<>();

    public Set<String> getChanged() {
        return ImmutableSet.<String>builder()
                .addAll(getAdded())
                .addAll(getUpdated())
                .addAll(getDeleted())
                .build();
    }

    public HashCode getChecksum(String path) {
        if (added.containsKey(path)) {
            return getAddedChecksum(path);
        }
        return getUpdatedChecksum(path);
    }

    public void added(String path, HashCode md5) {
        added.put(path, md5);
    }

    public Set<String> getAdded() {
        return added.keySet();
    }

    public HashCode getAddedChecksum(String path) {
        return added.get(path);
    }

    public Set<String> getUpdated() {
        return updated.keySet();
    }

    public HashCode getUpdatedChecksum(String path) {
        return updated.get(path);
    }

    public Set<String> getDeleted() {
        return deleted;
    }

    public void updated(String path, HashCode md5) {
        updated.put(path, md5);
    }

    public void deleted(String path) {
        deleted.add(path);
    }

    public boolean isAdded(String path) {
        return added.containsKey(path);
    }

    public boolean isUpdated(String path) {
        return updated.containsKey(path);
    }

    public boolean isDeleted(String path) {
        return deleted.contains(path);
    }

    public Optional<Change> changeFor(String path) {
        if (added.containsKey(path)) {
            return Optional.of(Change.ADDED);
        } else if (deleted.contains(path)) {
            return Optional.of(Change.DELETED);
        } else if (updated.containsKey(path)) {
            return Optional.of(Change.UPDATED);
        }
        return Optional.empty();
    }

    @Override
    public String location() {
        return "index";
    }

    @Override
    protected void writeToProperties(Properties properties) {
        writePart(properties, added.keySet(), "NEW", added);
        writePart(properties, updated.keySet(), "UPD", updated);
        writePart(properties, deleted, "DEL", null);
    }

    @Override
    protected void readFromProperties(Properties properties) {
        for (String path : properties.stringPropertyNames()) {
            String[] type = properties.getProperty(path).split("@");
            if ("NEW".equals(type[0])) {
                added(path, HashCode.fromString(type[1]));
            } else if ("UPD".equals(type[0])) {
                updated(path, HashCode.fromString(type[1]));
            } else if ("DEL".equals(type[0])) {
                deleted(path);
            }
        }
    }

    private void writePart(Properties props, Collection<String> paths, String type, Map<String, HashCode> checksums) {
        List<String> sorted = new ArrayList<>(paths);
        Collections.sort(sorted);

        for (String path : sorted) {
            HashCode md5 = null;
            if (checksums != null) {
                md5 = checksums.get(path);
            }
            props.setProperty(path, type + (md5 != null ? "@" + md5.toString() : ""));
        }
    }

    public boolean isEmpty() {
        return added.isEmpty() && updated.isEmpty() && deleted.isEmpty();
    }

    public enum Change {
        ADDED, UPDATED, DELETED
    }
}
