package com.suncode.autoupdate.patch.plusworkflow.archive;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import com.suncode.autoupdate.patch.PatchMeta;
import com.suncode.autoupdate.patch.plusworkflow.archive.PatchAssembler.Assemble;

/**
 * Binary patch archive.
 * 
 * @author Cezary Kozar 26 kwi 2016
 */
public class Archive
    implements AutoCloseable
{
    private final File source;

    private final boolean tempSource;

    private final Meta meta = new Meta();

    private final Index index = new Index();

    private final Checksum checksum = new Checksum();;

    private ZipFile zip;

    public Archive( File source )
    {
        this( source, false );
    }

    Archive( File source, boolean temporary )
    {
        this.source = source;
        this.tempSource = temporary;
    }

    public File getFile()
    {
        return source;
    }

    public Meta getMeta()
    {
        return meta;
    }

    public Index getIndex()
    {
        return index;
    }

    public Checksum getChecksum()
    {
        return checksum;
    }

    public void open()
        throws IOException
    {
        if ( this.zip == null )
        {
            this.zip = new ZipFile( source );
    
            readArchivePart( meta );
            readArchivePart( index );
            readArchivePart( checksum );
        }
    }

    public InputStream get( String path )
        throws IOException
    {
        final ZipEntry entry = zip.getEntry( "data/" + path );
        if(entry == null) {
            throw new IOException("Entry not found: " +  path);
        }
        return zip.getInputStream( entry );
    }

    private void readArchivePart( ArchivePart part )
        throws IOException
    {
        ZipEntry entry = zip.getEntry( part.location() );
        try (InputStream in = zip.getInputStream( entry );)
        {
            part.readFrom( in );
        }
    }

    @Override
    public void close()
        throws IOException
    {
        try
        {
            if ( zip != null )
            {
                zip.close();
            }
        }
        catch ( IOException e )
        {
            throw e;
        }
        finally
        {
            if ( tempSource )
            {
                source.delete();
            }
            zip = null;
        }
    }

    public static Archive assemble( PatchMeta patch, Assemble assemble )
        throws IOException
    {
        File storage = File.createTempFile( "patch-", ".zip" );
        try (PatchAssembler assembler = new PatchAssembler( patch, storage ))
        {
            assemble.assemble( assembler );
        }
        catch ( IOException | RuntimeException e )
        {
            storage.delete();
            throw e;
        }

        return new Archive( storage, true );
    }

    public static Archive assemble( PatchMeta patch, File storage, Assemble assemble )
        throws IOException
    {
        try (PatchAssembler assembler = new PatchAssembler( patch, storage ))
        {
            assemble.assemble( assembler );
        }
        return new Archive( storage );
    }

    public static PatchMeta readMeta(InputStream stream) {
        Meta meta = new Meta();
        ArchiveUtils.readArchivePart(stream, meta);
        return meta.get();
    }
}
