package com.suncode.pwfl.xpdl.builder;

import lombok.Getter;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.StringWriter;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

@Getter
class XpdlBuilder
    extends XpdlBaseBuilder
{

    private String packageId = "pakietxpdl";

    private String packageName = "PakietXPDL";

    private LocalDateTime created = LocalDateTime.now();

    private String author = "Administrator PlusWorkflow";

    private final List<XpdlProcessBuilder> processes = new ArrayList<>();

    public XpdlBuilder()
    {
        // TODO document why this constructor is empty
    }

    public XpdlBuilder withPackageId( String packageId )
    {
        this.packageId = packageId;
        return this;
    }

    public XpdlBuilder withPackageName( String packageName )
    {
        this.packageName = packageName;
        return this;
    }

    public XpdlBuilder withCreated( LocalDateTime created )
    {
        this.created = created;
        return this;
    }

    public XpdlBuilder withCreated( String created )
    {
        this.created = LocalDateTime.parse( created, DATE_TIME_FORMATTER );
        return this;
    }

    public XpdlBuilder withAuthor( String author )
    {
        this.author = author;
        return this;
    }

    public XpdlBuilder addProcess( XpdlProcessBuilder process )
    {
        processes.add( process );
        return this;
    }

    public XpdlProcessBuilder createProcess( String id, String name )
    {
        XpdlProcessBuilder process = XpdlProcessBuilder.create( id, name );
        processes.add( process );
        return process;
    }

    public XpdlProcessBuilder getFirstProcess()
    {
        if ( processes.isEmpty() )
        {
            throw new IllegalStateException( "No processes have been added. At least one process must be created." );
        }
        return processes.get( 0 );
    }

    public String build()
    {
        if ( processes.isEmpty() )
        {
            throw new IllegalStateException( "No processes have been added. At least one process must be created." );
        }
        Document document = createDocument();
        document.setXmlVersion( XML_VERSION );
        document.setXmlStandalone( true );

        Element rootElement = buildElement( document );
        document.appendChild( rootElement );

        return documentToString( document );
    }

    private Document createDocument()
    {
        try
        {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            return builder.newDocument();
        }
        catch ( ParserConfigurationException e )
        {
            throw new RuntimeException( "Error creating XML document", e );
        }
    }

    private String documentToString( Document document )
    {
        try
        {
            TransformerFactory transformerFactory = TransformerFactory.newInstance();
            Transformer transformer = transformerFactory.newTransformer();
            transformer.setOutputProperty( OutputKeys.INDENT, "yes" );
            transformer.setOutputProperty( "{http://xml.apache.org/xslt}indent-amount", "4" );
            transformer.setOutputProperty( OutputKeys.DOCTYPE_PUBLIC, "yes" );
            transformer.setOutputProperty( OutputKeys.ENCODING, "UTF-8" );

            DOMSource source = new DOMSource( document );

            StringWriter stringWriter = new StringWriter();
            StreamResult result = new StreamResult( stringWriter );

            transformer.transform( source, result );

            return stringWriter.toString();
        }
        catch ( TransformerException e )
        {
            throw new RuntimeException( "Error converting XML document to string", e );
        }
    }

    @Override
    public Element buildElement( Document document )
    {
        Element packageElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_PACKAGE );
        packageElement.setAttribute( "xmlns", XPDL_NAMESPACE );
        packageElement.setAttribute( "xmlns:xpdl", XPDL_NAMESPACE );
        packageElement.setAttribute( "xmlns:xsi", XSI_NAMESPACE );
        packageElement.setAttribute( ATTR_ID, packageId );
        packageElement.setAttribute( ATTR_NAME, packageName );
        packageElement.setAttribute( "xsi:schemaLocation",
                                     XPDL_NAMESPACE + " http://wfmc.org/standards/docs/TC-1025_schema_10_xpdl.xsd" );

        Element packageHeader = document.createElementNS( XPDL_NAMESPACE, ELEMENT_PACKAGE_HEADER );

        Element xpdlVersion = document.createElementNS( XPDL_NAMESPACE, ELEMENT_XPDL_VERSION );
        xpdlVersion.setTextContent( "1.0" );
        packageHeader.appendChild( xpdlVersion );

        Element vendor = document.createElementNS( XPDL_NAMESPACE, ELEMENT_VENDOR );
        vendor.setTextContent( "Suncode" );
        packageHeader.appendChild( vendor );

        Element createdElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_CREATED );
        createdElement.setTextContent( formatDateTime( created ) );
        packageHeader.appendChild( createdElement );

        packageElement.appendChild( packageHeader );

        Element redefinableHeader = document.createElementNS( XPDL_NAMESPACE, "xpdl:RedefinableHeader" );

        Element authorElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_AUTHOR );
        authorElement.setTextContent( author );
        redefinableHeader.appendChild( authorElement );

        packageElement.appendChild( redefinableHeader );

        Element script = document.createElementNS( XPDL_NAMESPACE, ELEMENT_SCRIPT );
        script.setAttribute( ATTR_TYPE, "text/javascript" );
        packageElement.appendChild( script );

        Element workflowProcesses = document.createElementNS( XPDL_NAMESPACE, ELEMENT_WORKFLOW_PROCESSES );

        for ( XpdlProcessBuilder process : processes )
        {
            workflowProcesses.appendChild( process.buildElement( document ) );
        }

        packageElement.appendChild( workflowProcesses );

        packageElement.appendChild( buildPackageExtendedAttributes( document ) );

        return packageElement;
    }

    private Element buildPackageExtendedAttributes( Document document )
    {
        Element extendedAttributes = document.createElementNS( XPDL_NAMESPACE, ELEMENT_EXTENDED_ATTRIBUTES );

        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_JAWE_CONFIGURATION, "default" );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_EDITING_TOOL, "Plus Workflow Editor" );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_EDITING_TOOL_VERSION, "2.2" );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_EVENT_ACTION_ID_COUNTER, "1" );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_PWE_VERSION, "4.2-SNAPSHOT" );

        return extendedAttributes;
    }

} 