package com.suncode.pwfl.assistant.phase;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.pwfl.assistant.AgentContext;
import com.suncode.pwfl.assistant.message.responses.LlmAnalysisResponse;
import com.suncode.pwfl.assistant.model.AgentTaskState;
import com.suncode.pwfl.assistant.model.Task;
import com.suncode.pwfl.assistant.agent.Agent;
import com.suncode.pwfl.prompting.LlmService;
import com.suncode.pwfl.util.StringUtils;
import lombok.extern.slf4j.Slf4j;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;

@Slf4j
public class AgentTaskPlanningPhase
{

    private final LlmService llmService;

    private final ObjectMapper objectMapper;

    public AgentTaskPlanningPhase( AgentContext context )
    {
        this.llmService = context.getLlmService();
        this.objectMapper = context.getObjectMapper();
    }

    public void execute(Agent agent, String userMessage) throws JsonProcessingException
    {
        String taskPrompt = agent.getTaskPrompt();

        String taskThoughtsRaw = llmService.chat(taskPrompt, userMessage);
        LlmAnalysisResponse<List<TaskAnalysis>> taskThoughtsAnalysis = objectMapper
            .readValue(taskThoughtsRaw, new TypeReference<>() {});

        log.trace("[{}][{}] Task planning phase completed: {}",
                  agent, agent.getState().getStep(),
                  taskThoughtsRaw);

        List<Task> updatedTasks = taskThoughtsAnalysis
            .getResult()
            .stream()
            .map(task -> {
                if (StringUtils.isNotBlank(task.uuid()))
                {
                    Optional<Task> existingTask = agent.getState()
                        .getTasks()
                        .stream()
                        .filter(alreadyExistingTask -> alreadyExistingTask.getUuid().equals(task.uuid()))
                        .findFirst();

                    if (existingTask.isPresent())
                    {
                        Task foundTask = existingTask.get();
                        if (foundTask.getState() == AgentTaskState.PENDING || 
                            foundTask.getState() == AgentTaskState.FAILED)
                        {
                            foundTask.setState(
                                StringUtils.isNotBlank( task.status() )
                                    ? AgentTaskState.valueOf( task.status().toUpperCase() )
                                    : foundTask.getState()
                            );
                            foundTask.setName(task.name());
                            foundTask.setDescription(task.description());
                            foundTask.setUpdated(LocalDateTime.now());
                        }
                        return foundTask;
                    }

                    log.warn( "[{}][{}] Task UUID provided but not found: {}",
                              agent, agent.getState().getStep(),
                              task.uuid() );
                    return null;
                }
                else
                {
                    return Task.builder()
                        .uuid(UUID.randomUUID().toString())
                        .conversationUuid(UUID.randomUUID().toString())
                        .state(AgentTaskState.PENDING)
                        .name(task.name())
                        .description(task.description())
                        .actions(new ArrayList<>())
                        .created(LocalDateTime.now())
                        .updated(LocalDateTime.now())
                        .failureCount(0)
                        .build();
                }
            })
            .filter(Objects::nonNull)
            .toList();

        agent.getState().setTasks( updatedTasks);

        agent.getState()
            .getTasks()
            .stream()
            .filter(task -> task.getState() == AgentTaskState.PENDING)
            .findFirst()
            .ifPresent( task -> agent.getState().setTask( task.getUuid() ) );
    }

    private record TaskAnalysis(String uuid, String name, String description, String status)
    {
    }

}
