package com.suncode.pwfl.assistant.phase;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.pwfl.assistant.AgentContext;
import com.suncode.pwfl.assistant.agent.Agent;
import com.suncode.pwfl.assistant.message.responses.ActionAnalysis;
import com.suncode.pwfl.assistant.message.responses.LlmAnalysisResponse;
import com.suncode.pwfl.assistant.model.Action;
import com.suncode.pwfl.assistant.model.AgentActionState;
import com.suncode.pwfl.assistant.model.Task;
import com.suncode.pwfl.prompting.LlmService;
import com.suncode.pwfl.util.StringUtils;
import lombok.extern.slf4j.Slf4j;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

@Slf4j
public class AgentActionPlanningPhase
{

    private final LlmService llmService;

    private final ObjectMapper objectMapper;

    public AgentActionPlanningPhase( AgentContext context )
    {
        this.llmService = context.getLlmService();
        this.objectMapper = context.getObjectMapper();
    }

    public void execute(Agent agent, String userMessage) throws JsonProcessingException
    {
        String actionPrompt = agent.getActionPrompt();

        String agentThoughtsRaw = llmService.chat(actionPrompt, userMessage);

        log.trace( "[{}][{}] Action planning phase completed: {}",
                   agent, agent.getState().getStep(),
                   agentThoughtsRaw );

        if (StringUtils.isBlank(agentThoughtsRaw))
        {
            return;
        }

        LlmAnalysisResponse<ActionAnalysis> actionAnalysisResponse = 
            objectMapper.readValue(agentThoughtsRaw, new TypeReference<>() {});

        ActionAnalysis actionAnalysis = actionAnalysisResponse.getResult();
        Optional<Task> taskToUpdateOptional = agent.getState()
            .getTasks()
            .stream()
            .filter(task -> task.getUuid().equals(actionAnalysis.getTaskUuid()))
            .findFirst();

        if (taskToUpdateOptional.isPresent())
        {
            Task taskToUpdate = taskToUpdateOptional.get();
            LocalDateTime now = LocalDateTime.now();
            
            Action action = Action.builder()
                .uuid(UUID.randomUUID().toString())
                .taskUuid(taskToUpdate.getUuid())
                .name(actionAnalysis.getName())
                .toolName(actionAnalysis.getToolName())
                .state(AgentActionState.PENDING)
                .created(now)
                .updated(now)
                .result(null)
                .build();

            taskToUpdate.setActions(List.of(action));
            agent.getState().setTask( taskToUpdate.getUuid());
            agent.getState().setAction( action.getUuid());
        }
        else
        {
            log.warn( "[{}][{}] Could not find task by uuid: {}",
                      agent, agent.getState().getStep(),
                      actionAnalysis.getTaskUuid() );
        }
    }

}
