package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.Variable;
import com.suncode.pwfl.workflow.process.map.VariableRef;
import com.suncode.pwfl.workflow.process.map.VariableType;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.text.Normalizer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Tool for updating variables in the process map.
 * Variables are used for storing data and displaying it on a form.
 * If a new name is provided, a new variableId is generated and all assignments are updated.
 */
@Getter
public class UpdateVariableTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "update_variable",
        """
            Updates one or more variables in the process map. Variables are used solely for storing data and displaying it on a form. For each variable, if a new name is provided, a new variableId is generated, and all assignments to activities are updated with the new variableId. Returns the updated variables in the following format: { "updatedVariables": [ { "oldVariableId": "<old id>", "newVariableId": "<new id>", "newVariableName": "<new name>", "newVariableType": "<new type>" } ] }
            """,
        "To update variables, write { \"variables\": [ { \"variableId\": \"<id to update>\", \"newVariableName\": \"<new name>\", \"newVariableType\": \"<new type>\" } ] }. The payload for each variable MUST be in the nominative case, for example: Invoice number, Invoice date, Invoice amount. Allowed variable types: 'STRING', 'INTEGER', 'FLOAT', 'BOOLEAN', 'DATE', 'DATE_TIME'",
        UpdateVariableTool.class
    );

    private final List<VariableToUpdate> variables;

    @JsonCreator
    public UpdateVariableTool(@JsonProperty("variables") List<VariableToUpdate> variables)
    {
        this.variables = variables;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (variables == null || variables.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"variables\" array."
            );
        }

        Process process = toolContext.getProcess();
        List<UpdatedVariableInfo> updatedVariables = new ArrayList<>();
        List<String> typeUpdateResults = new ArrayList<>();
        List<String> errors = new ArrayList<>();
        Map<String, String> idChanges = new HashMap<>();

        // First pass: validate and generate new IDs
        for (VariableToUpdate variableUpdate : variables)
        {
            Variable variable = process.getVariables().stream()
                .filter(v -> v.getId().equals(variableUpdate.getVariableId()))
                .findFirst()
                .orElse(null);

            if (variable == null)
            {
                errors.add("Error: Variable with variableId: '" + 
                          variableUpdate.getVariableId() + "' doesn't exist!.");
                continue;
            }

            if (variableUpdate.getNewVariableName() != null && 
                !variableUpdate.getNewVariableName().isBlank())
            {
                String newVariableId = generateUniqueVariableId(
                    process,
                    variableUpdate.getNewVariableName(),
                    variable.getId(),
                    idChanges
                );
                idChanges.put(variable.getId(), newVariableId);
            }
        }

        // Second pass: apply updates if no validation errors
        if (errors.isEmpty())
        {
            for (VariableToUpdate variableUpdate : variables)
            {
                Variable variable = process.getVariables().stream()
                    .filter(v -> v.getId().equals(variableUpdate.getVariableId()))
                    .findFirst()
                    .orElse(null);

                if (variable == null)
                {
                    continue;
                }

                String oldVariableId = variable.getId();
                int assignmentsUpdatedCount = 0;

                if (variableUpdate.getNewVariableName() != null && 
                    !variableUpdate.getNewVariableName().isBlank())
                {
                    String newVariableId = idChanges.get(oldVariableId);
                    variable.setName(variableUpdate.getNewVariableName());
                    variable.setId(newVariableId);

                    // Update all assignments to activities
                    for (Activity activity : process.getActivities())
                    {
                        for (VariableRef variableRef : activity.getVariableRefs())
                        {
                            if (variableRef.getId().equals(oldVariableId))
                            {
                                variableRef.setId(newVariableId);
                                assignmentsUpdatedCount++;
                            }
                        }
                    }

                    updatedVariables.add(new UpdatedVariableInfo(
                        oldVariableId,
                        newVariableId,
                        variableUpdate.getNewVariableName(),
                        variableUpdate.getNewVariableType() != null ? 
                            variableUpdate.getNewVariableType().name() : null,
                        assignmentsUpdatedCount
                    ));
                }

                if (variableUpdate.getNewVariableType() != null)
                {
                    variable.setType(variableUpdate.getNewVariableType());
                    if (variableUpdate.getNewVariableName() == null || 
                        variableUpdate.getNewVariableName().isBlank())
                    {
                        typeUpdateResults.add("Variable with variableId '" + oldVariableId + 
                                            "' updated successfully. New variableType is '" + 
                                            variableUpdate.getNewVariableType() + "'.");
                    }
                }
            }
        }

        String data = "";
        if (updatedVariables.size() > 0)
        {
            String formattedUpdates = updatedVariables.stream()
                .map(v -> "Variable updated. Old variableId: '" + v.oldVariableId() + 
                         "', New variableName: '" + v.newVariableName() + 
                         "', New variableId: '" + v.newVariableId() + "'. " +
                         (v.newVariableType() != null ? 
                             "New variableType: '" + v.newVariableType() + "'. " : "") +
                         "Updated " + v.assignmentsUpdatedCount() + " assignment(s).")
                .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);
            data += "Successfully updated " + updatedVariables.size() + " variables:\n" + 
                   formattedUpdates;
        }

        if (typeUpdateResults.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n";
            }
            data += String.join("\n", typeUpdateResults);
        }

        if (errors.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n\n";
            }
            data += "Encountered errors:\n" + String.join("\n", errors);
        }

        return new ToolResult(
            updatedVariables.size() > 0 || typeUpdateResults.size() > 0,
            data.isEmpty() ? "No variables were updated." : data
        );
    }

    /**
     * Generates a unique variable ID based on the variable name
     */
    private String generateUniqueVariableId(
        Process process,
        String variableName,
        String currentVariableId,
        Map<String, String> idChanges)
    {
        String baseVariableId = Normalizer.normalize(variableName, Normalizer.Form.NFD)
            .toLowerCase()
            .replaceAll("[\\u0300-\\u036f]", "")
            .replaceAll("\\s+", "_")
            .replaceAll("[^a-z0-9_]", "");
        
        if (baseVariableId.length() > 16)
        {
            baseVariableId = baseVariableId.substring(0, 16);
        }

        final String baseId = baseVariableId;
        String newVariableId = baseId;
        int counter = 1;

        while (true)
        {
            final String currentId = newVariableId;
            boolean isIdTaken = process.getVariables().stream()
                .anyMatch(v -> v.getId().equals(currentId) && 
                              !v.getId().equals(currentVariableId)) ||
                idChanges.values().stream().anyMatch(id -> id.equals(currentId));
            
            if (!isIdTaken)
            {
                break;
            }

            String counterStr = String.valueOf(counter);
            int maxLength = 16 - counterStr.length();
            String trimmedId = baseId.length() > maxLength ? 
                              baseId.substring(0, maxLength) : baseId;
            newVariableId = trimmedId + counter;
            counter++;
        }

        return newVariableId;
    }

    @Getter
    @Setter
    @AllArgsConstructor
    @NoArgsConstructor
    public static class VariableToUpdate
    {
        @JsonProperty(required = true)
        private String variableId;

        @JsonProperty
        private String newVariableName;

        @JsonProperty
        private VariableType newVariableType;
    }

    private record UpdatedVariableInfo(
        String oldVariableId,
        String newVariableId,
        String newVariableName,
        String newVariableType,
        int assignmentsUpdatedCount)
    {
    }
}
