package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.VariableRef;
import com.suncode.pwfl.workflow.process.map.VariableRefType;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Tool for updating variable assignments on a specific activity.
 * Allows changing the visibility type of assigned variables.
 */
@Getter
public class UpdateVariableAssignmentToActivityTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "update_variable_assignment_to_activity",
        "Updates one or more variable assignments on a specific activity. Returns the updated assignments in the following format: { \"results\": [ { \"activityId\": \"<id>\", \"variableId\": \"<id>\", \"visibilityType\": \"<type>\", \"status\": \"success|failed\", \"message\": \"<message>\" } ] }",
        "To update variable assignments on an activity, write { \"activityId\": \"<id>\", \"variables\": [ { \"variableId\": \"<id>\", \"visibilityType\": \"<type>\" } ] }. Allowed visibility types are 'UPDATE', 'VIEW', 'HIDDEN'. \nVisibility types description:\n- UPDATE: makes variable editable and visible\n- VIEW: makes variable not editable and visible\n- HIDDEN: makes variable editable and not visible",
        UpdateVariableAssignmentToActivityTool.class
    );

    private final String activityId;
    private final List<VariableRefToUpdate> variables;

    @JsonCreator
    public UpdateVariableAssignmentToActivityTool(
        @JsonProperty("activityId") String activityId,
        @JsonProperty("variables") List<VariableRefToUpdate> variables)
    {
        this.activityId = activityId;
        this.variables = variables;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityId == null || activityId.isBlank() || 
            variables == null || variables.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with \"activityId\" and a non-empty \"variables\" array."
            );
        }

        Process process = toolContext.getProcess();
        Optional<Activity> activityOpt = process.getActivities().stream()
            .filter(act -> act.getId().equals(activityId))
            .findFirst();

        if (activityOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Activity with activityId: '" + activityId + "' doesn't exist!."
            );
        }

        Activity activity = activityOpt.get();
        List<UpdateResult> results = new ArrayList<>();

        for (VariableRefToUpdate variableRefToUpdate : variables)
        {
            String variableId = variableRefToUpdate.getVariableId();
            VariableRefType visibilityType = variableRefToUpdate.getVariableRefType();

            Optional<VariableRef> variableRefOptional = activity.getVariableRefs().stream()
                .filter(variableRef -> variableRef.getId().equals(variableId))
                .findFirst();

            if (variableRefOptional.isEmpty())
            {
                results.add(new UpdateResult(
                    activityId,
                    variableId,
                    visibilityType.name(),
                    "failed",
                    "Error: Variable with variableId: '" + variableId + 
                    "' is not assigned to activity with activityId: '" + activityId + "'."
                ));
                continue;
            }

            VariableRef variableRef = variableRefOptional.get();
            variableRef.setType(visibilityType);
            results.add(new UpdateResult(
                activityId,
                variableId,
                visibilityType.name(),
                "success",
                "Successfully updated assignment of variable with variableId: '" + variableId + 
                "' on activity with activityId: '" + activityId + 
                "' to visibilityType: '" + visibilityType + "'."
            ));
        }

        String formattedData = results.stream()
            .map(r -> "activityId: " + r.activityId() + ", variableId: " + r.variableId() + 
                     ", status: " + r.status() + ", message: " + r.message())
            .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);

        boolean isSuccess = results.stream().anyMatch(r -> "success".equals(r.status()));

        return new ToolResult(isSuccess, "Update results:\n" + formattedData);
    }

    @Getter
    @Setter
    @AllArgsConstructor
    @NoArgsConstructor
    public static class VariableRefToUpdate
    {
        @JsonProperty(required = true)
        private String variableId;

        @JsonProperty(required = true)
        private VariableRefType variableRefType;
    }

    private record UpdateResult(
        String activityId,
        String variableId,
        String visibilityType,
        String status,
        String message)
    {
    }
}
