package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.transition.Transition;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.text.Normalizer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Tool for updating activity names in the process map.
 * A new activityId is generated for each activity based on its new name,
 * and all associated transitions are updated with the new activityId.
 */
@Getter
public class UpdateActivityTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "update_activity",
        """
            Updates one or more activities in the process map. A new activityId is generated for each activity based on its new name, and all associated transitions are updated with the new activityId. Returns the updated activities in the following format: { "updatedActivities": [ { "oldActivityId": "<old id>", "newActivityId": "<new id>", "newActivityName": "<new name>" } ] }
            """,
        "To update activities, write { \"activities\": [ { \"activityId\": \"<id to update>\", \"newActivityName\": \"<new name>\" } ] }.",
        UpdateActivityTool.class
    );

    private final List<ActivityNameUpdate> activities;

    @JsonCreator
    public UpdateActivityTool(@JsonProperty("activities") List<ActivityNameUpdate> activities)
    {
        this.activities = activities;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activities == null || activities.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"activities\" array."
            );
        }

        Process process = toolContext.getProcess();
        List<ActivityNameUpdateResult> updatedActivities = new ArrayList<>();
        List<String> errors = new ArrayList<>();
        Map<String, String> idChanges = new HashMap<>();

        // First pass: validate and generate new IDs
        for (ActivityNameUpdate activityUpdate : activities)
        {
            Activity activity = process.getActivities().stream()
                .filter(act -> act.getId().equals(activityUpdate.getActivityId()))
                .findFirst()
                .orElse(null);

            if (activity == null)
            {
                errors.add("Error: Activity with activityId: '" + activityUpdate.getActivityId() + 
                          "' doesn't exist!.");
                continue;
            }

            if (activityUpdate.getNewActivityName() == null || activityUpdate.getNewActivityName().isBlank())
            {
                errors.add("Error: newActivityName is required for activityId: '" + 
                          activityUpdate.getActivityId() + "'.");
                continue;
            }

            String newActivityId = generateUniqueActivityId(
                process, 
                activityUpdate.getNewActivityName(), 
                activity.getId(), 
                idChanges
            );
            idChanges.put(activity.getId(), newActivityId);
        }

        // Second pass: apply updates if no validation errors
        if (errors.isEmpty())
        {
            for (ActivityNameUpdate activityUpdate : activities)
            {
                Activity activity = process.getActivities().stream()
                    .filter(act -> act.getId().equals(activityUpdate.getActivityId()))
                    .findFirst()
                    .orElse(null);

                if (activity == null)
                {
                    continue;
                }

                String oldActivityId = activity.getId();
                String newActivityId = idChanges.get(oldActivityId);

                activity.setName(activityUpdate.getNewActivityName());
                activity.setId(newActivityId);

                updatedActivities.add(new ActivityNameUpdateResult(
                    oldActivityId, 
                    newActivityId, 
                    activityUpdate.getNewActivityName()
                ));
            }

            // Third pass: update all transitions
            for (Transition transition : process.getTransitions())
            {
                if (idChanges.containsKey(transition.getSourceId()))
                {
                    transition.setSourceId(idChanges.get(transition.getSourceId()));
                }

                if (idChanges.containsKey(transition.getTargetId()))
                {
                    transition.setTargetId(idChanges.get(transition.getTargetId()));
                }
            }
        }

        String data = "";
        if (updatedActivities.size() > 0)
        {
            String formattedUpdated = updatedActivities.stream()
                .map(u -> "Activity updated. Old activityId: '" + u.oldActivityId() + 
                         "', New activityName: '" + u.newActivityName() + 
                         "', New activityId: '" + u.newActivityId() + "'.")
                .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);
            data += "Successfully updated " + updatedActivities.size() + " activities:\n" + formattedUpdated;
        }

        if (errors.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n\n";
            }
            data += "Encountered errors:\n" + String.join("\n", errors);
        }

        return new ToolResult(
            updatedActivities.size() > 0,
            data.isEmpty() ? "No activities were updated." : data
        );
    }

    /**
     * Generates a unique activity ID based on the activity name
     */
    private String generateUniqueActivityId(
        Process process, 
        String activityName, 
        String currentActivityId,
        Map<String, String> idChanges)
    {
        String baseActivityId = Normalizer.normalize(activityName, Normalizer.Form.NFD)
            .toLowerCase()
            .replaceAll("[\\u0300-\\u036f]", "")
            .replaceAll("\\s+", "_")
            .replaceAll("[^a-z0-9_]", "");
        
        if (baseActivityId.length() > 16)
        {
            baseActivityId = baseActivityId.substring(0, 16);
        }

        final String baseId = baseActivityId;
        String newActivityId = baseId;
        int counter = 1;

        while (true)
        {
            final String currentId = newActivityId;
            boolean isIdTaken = process.getActivities().stream()
                .anyMatch(a -> a.getId().equals(currentId) && !a.getId().equals(currentActivityId)) ||
                idChanges.values().stream().anyMatch(id -> id.equals(currentId));
            
            if (!isIdTaken)
            {
                break;
            }

            String counterStr = String.valueOf(counter);
            int maxLength = 16 - counterStr.length();
            String trimmedId = baseId.length() > maxLength ? baseId.substring(0, maxLength) : baseId;
            newActivityId = trimmedId + counter;
            counter++;
        }

        return newActivityId;
    }

    @Getter
    @Setter
    @AllArgsConstructor
    @NoArgsConstructor
    public static class ActivityNameUpdate
    {
        @JsonProperty(required = true)
        private String activityId;

        @JsonProperty(required = true)
        private String newActivityName;
    }

    private record ActivityNameUpdateResult(String oldActivityId, String newActivityId, String newActivityName)
    {
    }
}
